/*
 * Copyright (c) 2003, 2026, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 *
 */

package sun.jvm.hotspot.debugger.linux.amd64;

import java.util.function.Function;

import sun.jvm.hotspot.debugger.*;
import sun.jvm.hotspot.debugger.amd64.*;
import sun.jvm.hotspot.debugger.linux.*;
import sun.jvm.hotspot.debugger.linux.amd64.*;
import sun.jvm.hotspot.debugger.cdbg.*;
import sun.jvm.hotspot.debugger.cdbg.basic.*;
import sun.jvm.hotspot.runtime.*;
import sun.jvm.hotspot.runtime.amd64.*;

public final class LinuxAMD64CFrame extends BasicCFrame {

   private static LinuxAMD64CFrame getFrameFromReg(LinuxDebugger dbg, Function<Integer, Address> getreg) {
      Address rip = getreg.apply(AMD64ThreadContext.RIP);
      Address rsp = getreg.apply(AMD64ThreadContext.RSP);
      Address rbp = getreg.apply(AMD64ThreadContext.RBP);
      Address libptr = dbg.findLibPtrByAddress(rip);
      Address cfa = null;
      DwarfParser dwarf = null;

      if (libptr != null) { // Native frame
        dwarf = new DwarfParser(libptr);
        try {
          dwarf.processDwarf(rip);
        } catch (DebuggerException e) {
          // DWARF processing should succeed when the frame is native
          // but it might fail if Common Information Entry (CIE) has language
          // personality routine and/or Language Specific Data Area (LSDA).
          return new LinuxAMD64CFrame(dbg, rsp, rbp, cfa, rip, dwarf, true);
        }

        cfa = getreg.apply(dwarf.getCFARegister())
                    .addOffsetTo(dwarf.getCFAOffset());
      }

      return (rbp == null && cfa == null)
        ? null
        : new LinuxAMD64CFrame(dbg, rsp, rbp, cfa, rip, dwarf);
   }

   public static LinuxAMD64CFrame getTopFrame(LinuxDebugger dbg, ThreadContext context) {
      return getFrameFromReg(dbg, context::getRegisterAsAddress);
   }

   private LinuxAMD64CFrame(LinuxDebugger dbg, Address rsp, Address rbp, Address cfa, Address rip, DwarfParser dwarf) {
      this(dbg, rsp, rbp, cfa, rip, dwarf, false);
   }

   private LinuxAMD64CFrame(LinuxDebugger dbg, Address rsp, Address rbp, Address cfa, Address rip, DwarfParser dwarf, boolean use1ByteBeforeToLookup) {
      super(dbg.getCDebugger());
      this.rsp = rsp;
      this.rbp = rbp;
      this.cfa = cfa;
      this.rip = rip;
      this.dbg = dbg;
      this.dwarf = dwarf;
      this.use1ByteBeforeToLookup = use1ByteBeforeToLookup;
   }

   // override base class impl to avoid ELF parsing
   public ClosestSymbol closestSymbolToPC() {
      Address symAddr = use1ByteBeforeToLookup ? pc().addOffsetTo(-1) : pc();
      var sym = dbg.lookup(dbg.getAddressValue(symAddr));

      // Returns a special symbol if the address is signal handler,
      // otherwise returns closest symbol generated by LinuxDebugger.
      return dbg.isSignalTrampoline(symAddr)
        ? new ClosestSymbol(sym.getName() + " <signal trampoline>", 0)
        : sym;
   }

   public Address pc() {
      return rip;
   }

   public Address localVariableBase() {
      return (dwarf != null && dwarf.isBPOffsetAvailable())
        ? cfa.addOffsetTo(dwarf.getBasePointerOffsetFromCFA())
        : rbp;
   }

   private Address getNextPC() {
     try {
       return dwarf == null
         ? rbp.getAddressAt(ADDRESS_SIZE) // Java frame
         : cfa.getAddressAt(dwarf.getReturnAddressOffsetFromCFA()); // Native frame
     } catch (UnmappedAddressException | UnalignedAddressException e) {
       return null;
     }
   }

   private boolean isValidFrame(Address nextCFA, Address nextRBP) {
     // Both CFA and RBP must not be null.
     if (nextCFA == null && nextRBP == null) {
       return false;
     }

     // RBP must not be null if CFA is null - it happens between Java frame and Native frame.
     // We cannot validate RBP value because it might be used as GPR. Thus returns true
     // if RBP is not null.
     if (nextCFA == null && nextRBP != null) {
       return true;
     }

     // nextCFA must be greater than current CFA.
     if (nextCFA != null && nextCFA.greaterThanOrEqual(cfa)) {
       return true;
     }

     // Otherwise, the frame is not valid.
     return false;
   }

   private Address getNextRSP() {
     return dwarf == null ? rbp.addOffsetTo(2 * ADDRESS_SIZE) // Java frame - skip saved BP and RA
                          : cfa.addOffsetTo(dwarf.getReturnAddressOffsetFromCFA())
                               .addOffsetTo(ADDRESS_SIZE); // Native frame
   }

   private Address getNextRBP(Address senderFP) {
     if (senderFP != null) {
       return senderFP;
     } else if (dwarf == null) { // Current frame is Java
       return rbp.getAddressAt(0);
     } else { // Current frame is Native
       return dwarf.isBPOffsetAvailable()
         ? cfa.getAddressAt(dwarf.getBasePointerOffsetFromCFA())
         : rbp;
     }
   }

   private Address getNextCFA(DwarfParser nextDwarf, Address senderFP, Address senderPC) {
     if (nextDwarf == null) { // Next frame is Java
       // CFA is not available on Java frame
       return null;
     }

     // Next frame is Native
     int nextCFAReg = nextDwarf.getCFARegister();
     return switch(nextCFAReg){
       case AMD64ThreadContext.RBP -> getNextRBP(senderFP).addOffsetTo(nextDwarf.getCFAOffset());
       case AMD64ThreadContext.RSP -> getNextRSP().addOffsetTo(nextDwarf.getCFAOffset());
       default -> throw new DebuggerException("Unsupported CFA register: " + nextCFAReg);
     };
   }

   @Override
   public CFrame sender(ThreadProxy th) {
     return sender(th, null, null, null);
   }

   @Override
   public CFrame sender(ThreadProxy th, Address sp, Address fp, Address pc) {
     if (dbg.isSignalTrampoline(pc())) {
       // RSP points signal context
       //   https://github.com/torvalds/linux/blob/v6.17/arch/x86/kernel/signal.c#L94
       return getFrameFromReg(dbg, r -> LinuxAMD64ThreadContext.getRegFromSignalTrampoline(this.rsp, r.intValue()));
     }

     ThreadContext context = th.getContext();

     Address nextRSP = sp != null ? sp : getNextRSP();
     if (nextRSP == null) {
       return null;
     }
     Address nextPC = pc != null ? pc : getNextPC();
     if (nextPC == null) {
       return null;
     }

     DwarfParser nextDwarf = null;
     boolean fallback = false;
     try {
       nextDwarf = createDwarfParser(nextPC);
     } catch (DebuggerException _) {
       // Try again with RIP-1 in case RIP is just outside function bounds,
       // due to function ending with a `call` instruction.
       try {
         nextDwarf = createDwarfParser(nextPC.addOffsetTo(-1));
         fallback = true;
       } catch (DebuggerException _) {
         // DWARF processing should succeed when the frame is native
         // but it might fail if Common Information Entry (CIE) has language
         // personality routine and/or Language Specific Data Area (LSDA).
         return null;
       }
     }

     Address nextRBP = getNextRBP(fp);

     try {
       Address nextCFA = getNextCFA(nextDwarf, fp, nextPC);
       return isValidFrame(nextCFA, nextRBP)
         ? new LinuxAMD64CFrame(dbg, nextRSP, nextRBP, nextCFA, nextPC, nextDwarf, fallback)
         : null;
     } catch (DebuggerException e) {
       if (dbg.isSignalTrampoline(nextPC)) {
         // We can through the caller frame if it is signal trampoline.
         // getNextCFA() might fail because DwarfParser cannot find out CFA register.
         return new LinuxAMD64CFrame(dbg, nextRSP, nextRBP, null, nextPC, nextDwarf, fallback);
       }

       // Rethrow the original exception if getNextCFA() failed
       // and the caller is not signal trampoline.
       throw e;
     }
   }

   private DwarfParser createDwarfParser(Address pc) throws DebuggerException {
     DwarfParser nextDwarf = null;
     Address libptr = dbg.findLibPtrByAddress(pc);
     if (libptr != null) {
       try {
         nextDwarf = new DwarfParser(libptr);
       } catch (DebuggerException _) {
         // Bail out to Java frame
       }
     }

     if (nextDwarf != null) {
       nextDwarf.processDwarf(pc);
     }

     return nextDwarf;
   }

   @Override
   public Frame toFrame() {
     return new AMD64Frame(rsp, localVariableBase(), rip);
   }

   // package/class internals only
   private static final int ADDRESS_SIZE = 8;
   private Address rsp;
   private Address rbp;
   private Address rip;
   private Address cfa;
   private LinuxDebugger dbg;
   private DwarfParser dwarf;
   private boolean use1ByteBeforeToLookup;
}
