package integration

import (
	"testing"

	"github.com/google/go-cmp/cmp"

	"github.com/anchore/syft/syft/pkg"
	"github.com/anchore/syft/syft/source"
)

func TestJavaPURLs(t *testing.T) {
	sbom, _ := catalogFixtureImage(t, "image-test-java-purls", source.SquashedScope)
	found := make(map[string]string)
	for _, p := range sbom.Artifacts.Packages.Sorted() {
		if p.Type != pkg.JavaPkg && p.Type != pkg.JenkinsPluginPkg {
			continue
		}

		if metadata, ok := p.Metadata.(pkg.JavaArchive); ok {
			if _, ok := noAssertion[metadata.VirtualPath]; ok {
				continue
			}

			found[metadata.VirtualPath] = p.PURL
		}
	}

	if d := cmp.Diff(expectedPURLs, found); d != "" {
		t.Errorf("unexpected purl values:\n%s", d)
	}
}

// This is for artifacts that either don't have a known groupid/artifactid
// since they aren't available on a known maven repository, or where syft is currently
// generating the incorrect PURL
var noAssertion = map[string]string{
	// These are not known to exist on any public maven repo, so no real PURL should be asserted
	"/packages/atlassian-bitbucket-server-integration.hpi:WEB-INF/lib/trilead-ssh2-build-217-jenkins-27.jar": "pkg:maven/trilead-ssh2-build/trilead-ssh2-build@217-jenkins-27",
	// possibly org.gradle as groupid for these, but unclear
	"/packages/gradle-7.1.1/lib/gradle-api-metadata-7.1.1.jar":                                                    "pkg:maven/gradle-api-metadata/gradle-api-metadata@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-base-annotations-7.1.1.jar":                                                "pkg:maven/gradle-base-annotations/gradle-base-annotations@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-base-services-7.1.1.jar":                                                   "pkg:maven/gradle-base-services/gradle-base-services@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-base-services-groovy-7.1.1.jar":                                            "pkg:maven/gradle-base-services-groovy/gradle-base-services-groovy@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-bootstrap-7.1.1.jar":                                                       "pkg:maven/gradle-bootstrap/gradle-bootstrap@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-build-cache-7.1.1.jar":                                                     "pkg:maven/gradle-build-cache/gradle-build-cache@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-build-cache-base-7.1.1.jar":                                                "pkg:maven/gradle-build-cache-base/gradle-build-cache-base@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-build-cache-packaging-7.1.1.jar":                                           "pkg:maven/gradle-build-cache-packaging/gradle-build-cache-packaging@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-build-events-7.1.1.jar":                                                    "pkg:maven/gradle-build-events/gradle-build-events@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-build-operations-7.1.1.jar":                                                "pkg:maven/gradle-build-operations/gradle-build-operations@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-build-option-7.1.1.jar":                                                    "pkg:maven/gradle-build-option/gradle-build-option@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-cli-7.1.1.jar":                                                             "pkg:maven/gradle-cli/gradle-cli@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-core-7.1.1.jar":                                                            "pkg:maven/gradle-core/gradle-core@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-core-api-7.1.1.jar":                                                        "pkg:maven/gradle-core-api/gradle-core-api@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-data-structures-7.1.1.jar":                                                 "pkg:maven/gradle-data-structures/gradle-data-structures@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-execution-7.1.1.jar":                                                       "pkg:maven/gradle-execution/gradle-execution@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-file-collections-7.1.1.jar":                                                "pkg:maven/gradle-file-collections/gradle-file-collections@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-file-temp-7.1.1.jar":                                                       "pkg:maven/gradle-file-temp/gradle-file-temp@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-file-watching-7.1.1.jar":                                                   "pkg:maven/gradle-file-watching/gradle-file-watching@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-files-7.1.1.jar":                                                           "pkg:maven/gradle-files/gradle-files@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-hashing-7.1.1.jar":                                                         "pkg:maven/gradle-hashing/gradle-hashing@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-installation-beacon-7.1.1.jar":                                             "pkg:maven/gradle-installation-beacon/gradle-installation-beacon@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-jvm-services-7.1.1.jar":                                                    "pkg:maven/gradle-jvm-services/gradle-jvm-services@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-kotlin-dsl-7.1.1.jar":                                                      "pkg:maven/gradle-kotlin-dsl/gradle-kotlin-dsl@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-kotlin-dsl-tooling-models-7.1.1.jar":                                       "pkg:maven/gradle-kotlin-dsl-tooling-models/gradle-kotlin-dsl-tooling-models@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-launcher-7.1.1.jar":                                                        "pkg:maven/org.gradle.launcher.GradleMain/gradle-launcher@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-logging-7.1.1.jar":                                                         "pkg:maven/gradle-logging/gradle-logging@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-messaging-7.1.1.jar":                                                       "pkg:maven/gradle-messaging/gradle-messaging@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-model-core-7.1.1.jar":                                                      "pkg:maven/gradle-model-core/gradle-model-core@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-model-groovy-7.1.1.jar":                                                    "pkg:maven/gradle-model-groovy/gradle-model-groovy@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-native-7.1.1.jar":                                                          "pkg:maven/gradle-native/gradle-native@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-normalization-java-7.1.1.jar":                                              "pkg:maven/gradle-normalization-java/gradle-normalization-java@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-persistent-cache-7.1.1.jar":                                                "pkg:maven/gradle-persistent-cache/gradle-persistent-cache@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-problems-7.1.1.jar":                                                        "pkg:maven/gradle-problems/gradle-problems@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-process-services-7.1.1.jar":                                                "pkg:maven/gradle-process-services/gradle-process-services@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-resources-7.1.1.jar":                                                       "pkg:maven/gradle-resources/gradle-resources@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-runtime-api-info-7.1.1.jar":                                                "pkg:maven/gradle-runtime-api-info/gradle-runtime-api-info@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-snapshots-7.1.1.jar":                                                       "pkg:maven/gradle-snapshots/gradle-snapshots@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-tooling-api-7.1.1.jar":                                                     "pkg:maven/gradle-tooling-api/gradle-tooling-api@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-worker-processes-7.1.1.jar":                                                "pkg:maven/gradle-worker-processes/gradle-worker-processes@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-wrapper-7.1.1.jar":                                                         "pkg:maven/gradle-wrapper/gradle-wrapper@7.1.1",
	"/packages/gradle-7.1.1/lib/gradle-wrapper-7.1.1.jar:gradle-wrapper.jar":                                      "pkg:maven/gradle-wrapper/gradle-wrapper",
	"/packages/gradle-7.1.1/lib/plugins/gradle-antlr-7.1.1.jar":                                                   "pkg:maven/gradle-antlr/gradle-antlr@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-build-cache-http-7.1.1.jar":                                        "pkg:maven/gradle-build-cache-http/gradle-build-cache-http@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-build-init-7.1.1.jar":                                              "pkg:maven/gradle-build-init/gradle-build-init@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-build-profile-7.1.1.jar":                                           "pkg:maven/gradle-build-profile/gradle-build-profile@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-code-quality-7.1.1.jar":                                            "pkg:maven/gradle-code-quality/gradle-code-quality@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-composite-builds-7.1.1.jar":                                        "pkg:maven/gradle-composite-builds/gradle-composite-builds@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-configuration-cache-7.1.1.jar":                                     "pkg:maven/gradle-configuration-cache/gradle-configuration-cache@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-dependency-management-7.1.1.jar":                                   "pkg:maven/gradle-dependency-management/gradle-dependency-management@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-diagnostics-7.1.1.jar":                                             "pkg:maven/gradle-diagnostics/gradle-diagnostics@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-ear-7.1.1.jar":                                                     "pkg:maven/gradle-ear/gradle-ear@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-enterprise-7.1.1.jar":                                              "pkg:maven/gradle-enterprise/gradle-enterprise@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-ide-7.1.1.jar":                                                     "pkg:maven/gradle-ide/gradle-ide@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-ide-native-7.1.1.jar":                                              "pkg:maven/gradle-ide-native/gradle-ide-native@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-ivy-7.1.1.jar":                                                     "pkg:maven/gradle-ivy/gradle-ivy@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-jacoco-7.1.1.jar":                                                  "pkg:maven/gradle-jacoco/gradle-jacoco@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-java-compiler-plugin-7.1.1.jar":                                    "pkg:maven/gradle-java-compiler-plugin/gradle-java-compiler-plugin@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-kotlin-dsl-provider-plugins-7.1.1.jar":                             "pkg:maven/gradle-kotlin-dsl-provider-plugins/gradle-kotlin-dsl-provider-plugins@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-kotlin-dsl-tooling-builders-7.1.1.jar":                             "pkg:maven/gradle-kotlin-dsl-tooling-builders/gradle-kotlin-dsl-tooling-builders@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-language-groovy-7.1.1.jar":                                         "pkg:maven/gradle-language-groovy/gradle-language-groovy@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-language-java-7.1.1.jar":                                           "pkg:maven/gradle-language-java/gradle-language-java@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-language-jvm-7.1.1.jar":                                            "pkg:maven/gradle-language-jvm/gradle-language-jvm@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-language-native-7.1.1.jar":                                         "pkg:maven/gradle-language-native/gradle-language-native@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-maven-7.1.1.jar":                                                   "pkg:maven/gradle-maven/gradle-maven@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-platform-base-7.1.1.jar":                                           "pkg:maven/gradle-platform-base/gradle-platform-base@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-platform-jvm-7.1.1.jar":                                            "pkg:maven/gradle-platform-jvm/gradle-platform-jvm@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-platform-native-7.1.1.jar":                                         "pkg:maven/gradle-platform-native/gradle-platform-native@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-plugin-development-7.1.1.jar":                                      "pkg:maven/gradle-plugin-development/gradle-plugin-development@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-plugin-use-7.1.1.jar":                                              "pkg:maven/gradle-plugin-use/gradle-plugin-use@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-plugins-7.1.1.jar":                                                 "pkg:maven/gradle-plugins/gradle-plugins@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-publish-7.1.1.jar":                                                 "pkg:maven/gradle-publish/gradle-publish@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-reporting-7.1.1.jar":                                               "pkg:maven/gradle-reporting/gradle-reporting@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-resources-gcs-7.1.1.jar":                                           "pkg:maven/gradle-resources-gcs/gradle-resources-gcs@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-resources-http-7.1.1.jar":                                          "pkg:maven/gradle-resources-http/gradle-resources-http@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-resources-s3-7.1.1.jar":                                            "pkg:maven/gradle-resources-s3/gradle-resources-s3@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-resources-sftp-7.1.1.jar":                                          "pkg:maven/gradle-resources-sftp/gradle-resources-sftp@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-scala-7.1.1.jar":                                                   "pkg:maven/gradle-scala/gradle-scala@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-security-7.1.1.jar":                                                "pkg:maven/gradle-security/gradle-security@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-signing-7.1.1.jar":                                                 "pkg:maven/gradle-signing/gradle-signing@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-test-kit-7.1.1.jar":                                                "pkg:maven/gradle-test-kit/gradle-test-kit@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-testing-base-7.1.1.jar":                                            "pkg:maven/gradle-testing-base/gradle-testing-base@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-testing-junit-platform-7.1.1.jar":                                  "pkg:maven/gradle-testing-junit-platform/gradle-testing-junit-platform@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-testing-jvm-7.1.1.jar":                                             "pkg:maven/gradle-testing-jvm/gradle-testing-jvm@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-testing-native-7.1.1.jar":                                          "pkg:maven/gradle-testing-native/gradle-testing-native@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-tooling-api-builders-7.1.1.jar":                                    "pkg:maven/gradle-tooling-api-builders/gradle-tooling-api-builders@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-tooling-native-7.1.1.jar":                                          "pkg:maven/gradle-tooling-native/gradle-tooling-native@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-version-control-7.1.1.jar":                                         "pkg:maven/gradle-version-control/gradle-version-control@7.1.1",
	"/packages/gradle-7.1.1/lib/plugins/gradle-workers-7.1.1.jar":                                                 "pkg:maven/gradle-workers/gradle-workers@7.1.1",
	"/packages/jruby-stdlib-9.1.15.0.jar:META-INF/jruby.home/lib/ruby/stdlib/jopenssl.jar":                        "pkg:maven/jopenssl/jopenssl",
	"/packages/jruby-stdlib-9.1.15.0.jar:META-INF/jruby.home/lib/ruby/stdlib/jopenssl.jar:rubygems:jruby-openssl": "pkg:maven/rubygems/jruby-openssl@0.9.21",
	"/packages/jruby-stdlib-9.1.15.0.jar:META-INF/jruby.home/lib/ruby/stdlib/json/ext/parser.jar":                 "pkg:maven/parser/parser",
	"/packages/jruby-stdlib-9.1.15.0.jar:META-INF/jruby.home/lib/ruby/stdlib/json/ext/generator.jar":              "pkg:maven/generator/generator",
	"/packages/jruby-stdlib-9.1.15.0.jar:META-INF/jruby.home/lib/ruby/stdlib/psych.jar":                           "pkg:maven/psych/psych",
	"/packages/jruby-stdlib-9.1.15.0.jar:META-INF/jruby.home/lib/ruby/stdlib/racc/cparse-jruby.jar":               "pkg:maven/cparse-jruby/cparse-jruby",
	"/packages/jruby-stdlib-9.1.15.0.jar:META-INF/jruby.home/lib/ruby/stdlib/readline.jar":                        "pkg:maven/readline/readline",
	"/packages/org.eclipse.ant.core-3.7.0.jar:lib/antsupportlib.jar":                                              "pkg:maven/antsupportlib/antsupportlib",
	"/packages/TwilioNotifier.hpi:WEB-INF/lib/sdk-3.0.jar":                                                        "pkg:maven/sdk/sdk@3.0",

	// syft generates incorrect purls
	"/packages/atlassian-bitbucket-server-integration.hpi:WEB-INF/lib/jbcrypt-1.0.0.jar":        "pkg:maven/org.connectbot/jbcrypt@1.0.0",
	"/packages/atlassian-bitbucket-server-integration.hpi:WEB-INF/lib/tink-1.5.0.jar":           "pkg:maven/com.google.crypto.tink/tink@1.5.0",
	"/packages/gradle-7.1.1/lib/fastutil-8.5.2-min.jar":                                         "pkg:maven/it.unimi.dsi/fastutil@8.5.2-min",
	"/packages/gradle-7.1.1/lib/file-events-0.22-milestone-16.jar":                              "pkg:maven/net.rubygrapefruit/file-events@0.22-milestone-16",
	"/packages/gradle-7.1.1/lib/file-events-linux-aarch64-0.22-milestone-16.jar":                "pkg:maven/net.rubygrapefruit/file-events-linux-aarch64@0.22-milestone-16",
	"/packages/gradle-7.1.1/lib/file-events-linux-amd64-0.22-milestone-16.jar":                  "pkg:maven/net.rubygrapefruit/file-events-linux-amd64@0.22-milestone-16",
	"/packages/gradle-7.1.1/lib/file-events-osx-aarch64-0.22-milestone-16.jar":                  "pkg:maven/net.rubygrapefruit/file-events-osx-aarch64@0.22-milestone-16",
	"/packages/gradle-7.1.1/lib/file-events-osx-amd64-0.22-milestone-16.jar":                    "pkg:maven/net.rubygrapefruit/file-events-osx-amd64@0.22-milestone-16",
	"/packages/gradle-7.1.1/lib/file-events-windows-amd64-0.22-milestone-16.jar":                "pkg:maven/net.rubygrapefruit/file-events-windows-amd64@0.22-milestone-16",
	"/packages/gradle-7.1.1/lib/file-events-windows-amd64-min-0.22-milestone-16.jar":            "pkg:maven/net.rubygrapefruit/file-events-windows-amd64-min@0.22-milestone-16",
	"/packages/gradle-7.1.1/lib/file-events-windows-i386-0.22-milestone-16.jar":                 "pkg:maven/net.rubygrapefruit/file-events-windows-i386@0.22-milestone-16",
	"/packages/gradle-7.1.1/lib/file-events-windows-i386-min-0.22-milestone-16.jar":             "pkg:maven/net.rubygrapefruit/file-events-windows-i386-min@0.22-milestone-16",
	"/packages/gradle-7.1.1/lib/hamcrest-core-1.3.jar":                                          "pkg:maven/org.hamcrest/hamcrest-core@1.3",
	"/packages/gradle-7.1.1/lib/kotlin-compiler-embeddable-1.4.31-patched-for-gradle-7.1.1.jar": "pkg:maven/org.jetbrains.kotlin/kotlin-compiler-embeddable@1.4.31-patched-for-gradle-7.1.1",
	"/packages/gradle-7.1.1/lib/kotlin-daemon-embeddable-1.4.31.jar":                            "pkg:maven/org.jetbrains.kotlin/kotlin-daemon-embeddable@1.4.31",
	"/packages/gradle-7.1.1/lib/kotlin-reflect-1.4.31.jar":                                      "pkg:maven/org.jetbrains.kotlin/kotlin-reflect@1.4.31",
	"/packages/gradle-7.1.1/lib/kotlin-sam-with-receiver-compiler-plugin-1.4.31.jar":            "pkg:maven/org.jetbrains.kotlin/kotlin-sam-with-receiver-compiler-plugin@1.4.31",
	"/packages/gradle-7.1.1/lib/kotlin-script-runtime-1.4.31.jar":                               "pkg:maven/org.jetbrains.kotlin/kotlin-script-runtime@1.4.31",
	"/packages/gradle-7.1.1/lib/kotlin-scripting-common-1.4.31.jar":                             "pkg:maven/org.jetbrains.kotlin/kotlin-scripting-common@1.4.31",
	"/packages/gradle-7.1.1/lib/kotlin-scripting-compiler-embeddable-1.4.31.jar":                "pkg:maven/org.jetbrains.kotlin/kotlin-scripting-compiler-embeddable@1.4.31",
	"/packages/gradle-7.1.1/lib/kotlin-scripting-compiler-impl-embeddable-1.4.31.jar":           "pkg:maven/org.jetbrains.kotlin/kotlin-scripting-compiler-impl-embeddable@1.4.31",
	"/packages/gradle-7.1.1/lib/kotlin-scripting-jvm-1.4.31.jar":                                "pkg:maven/org.jetbrains.kotlin/kotlin-scripting-jvm@1.4.31",
	"/packages/gradle-7.1.1/lib/kotlin-scripting-jvm-host-1.4.31.jar":                           "pkg:maven/org.jetbrains.kotlin/kotlin-scripting-jvm-host@1.4.31",
	"/packages/gradle-7.1.1/lib/kotlin-stdlib-1.4.31.jar":                                       "pkg:maven/org.jetbrains.kotlin/kotlin-stdlib@1.4.31",
	"/packages/gradle-7.1.1/lib/kotlin-stdlib-common-1.4.31.jar":                                "pkg:maven/org.jetbrains.kotlin/kotlin-stdlib-common@1.4.31",
	"/packages/gradle-7.1.1/lib/kotlin-stdlib-jdk7-1.4.31.jar":                                  "pkg:maven/org.jetbrains.kotlin/kotlin-stdlib-jdk7@1.4.31",
	"/packages/gradle-7.1.1/lib/kotlin-stdlib-jdk8-1.4.31.jar":                                  "pkg:maven/org.jetbrains.kotlin/kotlin-stdlib-jdk8@1.4.31",
	"/packages/gradle-7.1.1/lib/kotlinx-metadata-jvm-0.2.0.jar":                                 "pkg:maven/org.jetbrains.kotlinx/kotlinx-metadata-jvm@0.2.0",
	"/packages/gradle-7.1.1/lib/native-platform-0.22-milestone-16.jar":                          "pkg:maven/net.rubygrapefruit/native-platform@0.22-milestone-16",
	"/packages/gradle-7.1.1/lib/native-platform-freebsd-amd64-libcpp-0.22-milestone-16.jar":     "pkg:maven/net.rubygrapefruit/native-platform-freebsd-amd64-libcpp@0.22-milestone-16",
	"/packages/gradle-7.1.1/lib/native-platform-linux-aarch64-0.22-milestone-16.jar":            "pkg:maven/net.rubygrapefruit/native-platform-linux-aarch64@0.22-milestone-16",
	"/packages/gradle-7.1.1/lib/native-platform-linux-aarch64-ncurses5-0.22-milestone-16.jar":   "pkg:maven/net.rubygrapefruit/native-platform-linux-aarch64-ncurses5@0.22-milestone-16",
	"/packages/gradle-7.1.1/lib/native-platform-linux-aarch64-ncurses6-0.22-milestone-16.jar":   "pkg:maven/net.rubygrapefruit/native-platform-linux-aarch64-ncurses6@0.22-milestone-16",
	"/packages/gradle-7.1.1/lib/native-platform-linux-amd64-0.22-milestone-16.jar":              "pkg:maven/net.rubygrapefruit/native-platform-linux-amd64@0.22-milestone-16",
	"/packages/gradle-7.1.1/lib/native-platform-linux-amd64-ncurses5-0.22-milestone-16.jar":     "pkg:maven/net.rubygrapefruit/native-platform-linux-amd64-ncurses5@0.22-milestone-16",
	"/packages/gradle-7.1.1/lib/native-platform-linux-amd64-ncurses6-0.22-milestone-16.jar":     "pkg:maven/net.rubygrapefruit/native-platform-linux-amd64-ncurses6@0.22-milestone-16",
	"/packages/gradle-7.1.1/lib/native-platform-osx-aarch64-0.22-milestone-16.jar":              "pkg:maven/net.rubygrapefruit/native-platform-osx-aarch64@0.22-milestone-16",
	"/packages/gradle-7.1.1/lib/native-platform-osx-amd64-0.22-milestone-16.jar":                "pkg:maven/net.rubygrapefruit/native-platform-osx-amd64@0.22-milestone-16",
	"/packages/gradle-7.1.1/lib/native-platform-windows-amd64-0.22-milestone-16.jar":            "pkg:maven/net.rubygrapefruit/native-platform-windows-amd64@0.22-milestone-16",
	"/packages/gradle-7.1.1/lib/native-platform-windows-amd64-min-0.22-milestone-16.jar":        "pkg:maven/net.rubygrapefruit/native-platform-windows-amd64-min@0.22-milestone-16",
	"/packages/gradle-7.1.1/lib/native-platform-windows-i386-0.22-milestone-16.jar":             "pkg:maven/net.rubygrapefruit/native-platform-windows-i386@0.22-milestone-16",
	"/packages/gradle-7.1.1/lib/native-platform-windows-i386-min-0.22-milestone-16.jar":         "pkg:maven/net.rubygrapefruit/native-platform-windows-i386-min@0.22-milestone-16",
	"/packages/gradle-7.1.1/lib/plugins/apiguardian-api-1.1.0.jar":                              "pkg:maven/org.apiguardian/apiguardian-api@1.1.0",
	"/packages/gradle-7.1.1/lib/plugins/bsh-2.0b6.jar":                                          "pkg:maven/org.apache-extras.beanshell/bsh@2.0b6",
	"/packages/gradle-7.1.1/lib/plugins/google-api-client-1.25.0.jar":                           "pkg:maven/com.google.api-client/google-api-client@1.25.0",
	"/packages/gradle-7.1.1/lib/plugins/google-api-services-storage-v1-rev171-1.25.0.jar":       "pkg:maven/com.google.apis/google-api-services-storage@v1-rev171-1.25.0",
	"/packages/gradle-7.1.1/lib/plugins/jcommander-1.78.jar":                                    "pkg:maven/com.beust/jcommander@1.78",
	"/packages/gradle-7.1.1/lib/plugins/junit-platform-commons-1.7.2.jar":                       "pkg:maven/org.junit.platform/junit-platform-commons@1.7.2",
	"/packages/gradle-7.1.1/lib/plugins/junit-platform-engine-1.7.2.jar":                        "pkg:maven/org.junit.platform/junit-platform-engine@1.7.2",
	"/packages/gradle-7.1.1/lib/plugins/junit-platform-launcher-1.7.2.jar":                      "pkg:maven/org.junit.platform/junit-platform-launcher@1.7.2",
	"/packages/gradle-7.1.1/lib/plugins/nekohtml-1.9.22.jar":                                    "pkg:maven/net.sourceforge.nekohtml/nekohtml@1.9.22",
	"/packages/gradle-7.1.1/lib/tomlj-1.0.0.jar":                                                "pkg:maven/org.tomlj/tomlj@1.0.0",
	"/packages/gradle-7.1.1/lib/trove4j-1.0.20181211.jar":                                       "pkg:maven/net.sf.trove4j /trove4j@1.0.20181211",
	"/packages/gradle-7.1.1/lib/xml-apis-1.4.01.jar":                                            "pkg:maven/xml-apis/xml-apis@1.4.01",
	"/packages/gradle-7.1.1/lib/asm-analysis-9.1.jar":                                           "pkg:maven/org.ow2.asm/asm-analysis@9.1",
	"/packages/gradle-7.1.1/lib/asm-commons-9.1.jar":                                            "pkg:maven/org.ow2.asm/asm-commons@9.1",
	"/packages/gradle-7.1.1/lib/asm-tree-9.1.jar":                                               "pkg:maven/org.ow2.asm/asm-tree@9.1",
	"/packages/gradle-7.1.1/lib/plugins/ivy-2.3.0.jar":                                          "pkg:maven/org.apache.ivy/ivy@2.3.0",
	"/packages/hudson.war:WEB-INF/lib/asm-2.2.3.jar":                                            "pkg:maven/asm/asm@2.2.3",
	"/packages/hudson.war:WEB-INF/lib/asm-commons-2.2.3.jar":                                    "pkg:maven/asm/asm-commons@2.2.3",
	"/packages/hudson.war:WEB-INF/lib/asm-tree-2.2.3.jar":                                       "pkg:maven/asm/asm-tree@2.2.3",
	"/packages/hudson.war:WEB-INF/slave.jar":                                                    "pkg:maven/org.jvnet.hudson.main/remoting@1.390",
	"/packages/hudson.war:WEB-INF/lib/xpp3_min-1.1.4c.jar":                                      "pkg:maven/xpp3_min/xpp3_min@1.1.4c",
	"/packages/hudson.war:WEB-INF/lib/xpp3-1.1.4c.jar":                                          "pkg:maven/xpp3/xpp3@1.1.4c",
	"/packages/hudson.war:WEB-INF/hudson-cli.jar":                                               "pkg:maven/org.jvnet.hudson.main/hudson-cli@1.390",
	"/packages/hudson.war:WEB-INF/lib/dom4j-1.6.1-hudson-3.jar":                                 "pkg:maven/org.jvnet.hudson.dom4j/dom4j@1.6.1-hudson-3",
	"/packages/newrelic-agent-8.8.0.jar":                                                        "pkg:maven/com.newrelic.bootstrap.BootstrapAgent/newrelic-agent@8.8.0",
	"/packages/newrelic-agent-8.8.0.jar:agent-bridge-datastore.jar":                             "pkg:maven/agent-bridge-datastore/agent-bridge-datastore@8.8.0",
	"/packages/newrelic-agent-8.8.0.jar:agent-bridge.jar":                                       "pkg:maven/agent-bridge/agent-bridge@8.8.0",
	"/packages/newrelic-agent-8.8.0.jar:newrelic-api.jar":                                       "pkg:maven/newrelic-api/newrelic-api@8.8.0",
	"/packages/newrelic-agent-8.8.0.jar:newrelic-security-agent.jar":                            "pkg:maven/newrelic-security-agent/newrelic-security-agent@1.0.7-public-preview",
	"/packages/newrelic-agent-8.8.0.jar:newrelic-security-api.jar":                              "pkg:maven/newrelic-security-api/newrelic-security-api@1.0.7-public-preview",
	"/packages/newrelic-agent-8.8.0.jar:newrelic-weaver-api.jar":                                "pkg:maven/newrelic-weaver-api/newrelic-weaver-api@8.8.0",
	"/packages/newrelic-agent-8.8.0.jar:newrelic-weaver-scala-api.jar":                          "pkg:maven/newrelic-weaver-scala-api/newrelic-weaver-scala-api@8.8.0",
	"/packages/maven-plugin.hpi:WEB-INF/lib/jsr250-api-1.0.jar":                                 "pkg:maven/javax.annotation/jsr250-api@1.0",
	"/packages/TwilioNotifier.hpi:WEB-INF/lib/commons-httpclient-3.1.jar":                       "pkg:maven/commons-httpclient/commons-httpclient@3.1",
	"/packages/xpp3_min-1.1.4c.jar":                                                             "pkg:maven/xpp3/xpp3_min@1.1.4c",

	// Unclear: groupid might be org.glassfish.jaxb or com.sun.xml.bind
	"/packages/gradle-7.1.1/lib/plugins/jaxb-core-3.0.0.jar": "pkg:maven/org.glassfish.jaxb/jaxb-core@3.0.0",
}

// Constructed by:
// syft anchore/test_images:java-1abc58f -o json | jq -r '.artifacts.[] | [.metadata.virtualPath, .purl, ""] | @csv' | grep 'pkg:maven' | sort | uniq >> /tmp/java_artifacts_mapping.txt
// The map was then hand-edited for correctness by comparing to Maven Central.
var expectedPURLs = map[string]string{
	"/packages/activemq-client-5.18.2.jar":                       "pkg:maven/org.apache.activemq/activemq-client@5.18.2",
	"/packages/activemq-protobuf-1.1.jar":                        "pkg:maven/org.apache.activemq.protobuf/activemq-protobuf@1.1",
	"/packages/akka-actor_2.13-2.6.6.jar":                        "pkg:maven/com.typesafe.akka/akka-actor_2.13@2.6.6",
	"/packages/akka-management-cluster-bootstrap_2.13-1.2.0.jar": "pkg:maven/com.lightbend.akka.management/akka-management-cluster-bootstrap_2.13@1.2.0",
	"/packages/ant-1.10.3.jar":                                   "pkg:maven/org.apache.ant/ant@1.10.3",
	"/packages/apache-chainsaw-2.1.0.jar":                        "pkg:maven/log4j/apache-chainsaw@2.1.0",
	"/packages/apache-log4j-extras-1.1.jar":                      "pkg:maven/log4j/apache-log4j-extras@1.1",
	"/packages/apoc-4.4.0.11.jar":                                "pkg:maven/org.neo4j.procedure/apoc@4.4.0.11",
	"/packages/atlassian-bitbucket-server-integration.hpi":       "pkg:maven/io.jenkins.plugins/atlassian-bitbucket-server-integration@2.1.2",
	"/packages/atlassian-bitbucket-server-integration.hpi:WEB-INF/lib/atlassian-bitbucket-server-integration.jar":                        "pkg:maven/io.jenkins.plugins/atlassian-bitbucket-server-integration@2.1.2",
	"/packages/atlassian-bitbucket-server-integration.hpi:WEB-INF/lib/commons-codec-1.13.jar":                                            "pkg:maven/commons-codec/commons-codec@1.13",
	"/packages/atlassian-bitbucket-server-integration.hpi:WEB-INF/lib/commons-lang3-3.9.jar":                                             "pkg:maven/org.apache.commons/commons-lang3@3.9",
	"/packages/atlassian-bitbucket-server-integration.hpi:WEB-INF/lib/eddsa-0.3.0.jar":                                                   "pkg:maven/net.i2p.crypto/eddsa@0.3.0",
	"/packages/atlassian-bitbucket-server-integration.hpi:WEB-INF/lib/httpcore-4.4.9.jar":                                                "pkg:maven/org.apache.httpcomponents/httpcore@4.4.9",
	"/packages/atlassian-bitbucket-server-integration.hpi:WEB-INF/lib/jackson-annotations-2.11.2.jar":                                    "pkg:maven/com.fasterxml.jackson.core/jackson-annotations@2.11.2",
	"/packages/atlassian-bitbucket-server-integration.hpi:WEB-INF/lib/jackson-core-2.11.2.jar":                                           "pkg:maven/com.fasterxml.jackson.core/jackson-core@2.11.2",
	"/packages/atlassian-bitbucket-server-integration.hpi:WEB-INF/lib/jackson-databind-2.9.10.7.jar":                                     "pkg:maven/com.fasterxml.jackson.core/jackson-databind@2.9.10.7",
	"/packages/atlassian-bitbucket-server-integration.hpi:WEB-INF/lib/jcommon-1.0.24.jar":                                                "pkg:maven/org.jfree/jcommon@1.0.24",
	"/packages/atlassian-bitbucket-server-integration.hpi:WEB-INF/lib/json-20180813.jar":                                                 "pkg:maven/org.json/json@20180813",
	"/packages/atlassian-bitbucket-server-integration.hpi:WEB-INF/lib/oauth-20100527.jar":                                                "pkg:maven/net.oauth.core/oauth@20100527",
	"/packages/atlassian-bitbucket-server-integration.hpi:WEB-INF/lib/oauth-provider-20100527.jar":                                       "pkg:maven/net.oauth.core/oauth-provider@20100527",
	"/packages/atlassian-bitbucket-server-integration.hpi:WEB-INF/lib/okhttp-3.14.1.jar":                                                 "pkg:maven/com.squareup.okhttp3/okhttp@3.14.1",
	"/packages/atlassian-bitbucket-server-integration.hpi:WEB-INF/lib/okio-1.17.2.jar":                                                   "pkg:maven/com.squareup.okio/okio@1.17.2",
	"/packages/atlassian-bitbucket-server-integration.hpi:WEB-INF/lib/protobuf-java-3.11.1.jar":                                          "pkg:maven/com.google.protobuf/protobuf-java@3.11.1",
	"/packages/atlassian-bitbucket-server-integration.hpi:WEB-INF/lib/trilead-putty-extension-1.2.jar":                                   "pkg:maven/org.kohsuke/trilead-putty-extension@1.2",
	"/packages/atlassian-bitbucket-server-integration.hpi:WEB-INF/lib/trilead-ssh2-build-217-jenkins-27.jar:org.jenkins-ci:trilead-ssh2": "pkg:maven/org.jenkins-ci/trilead-ssh2@build-217-jenkins-27",
	"/packages/aws-sam.hpi":                                                                                                           "pkg:maven/io.jenkins.plugins/aws-sam@1.2.2",
	"/packages/aws-sam.hpi:WEB-INF/lib/aws-sam.jar":                                                                                   "pkg:maven/io.jenkins.plugins/aws-sam@1.2.2",
	"/packages/aws-sam.hpi:WEB-INF/lib/json-20180130.jar":                                                                             "pkg:maven/org.json/json@20180130",
	"/packages/aws-sam.hpi:WEB-INF/lib/snakeyaml-1.21.jar":                                                                            "pkg:maven/org.yaml/snakeyaml@1.21",
	"/packages/bc-fips-1.0.2.3.jar":                                                                                                   "pkg:maven/org.bouncycastle/bc-fips@1.0.2.3",
	"/packages/camel-core-3.1.0.jar":                                                                                                  "pkg:maven/org.apache.camel/camel-core@3.1.0",
	"/packages/cassandra-all-4.1.1.jar":                                                                                               "pkg:maven/org.apache.cassandra/cassandra-all@4.1.1",
	"/packages/commons-logging-1.1.1.jar":                                                                                             "pkg:maven/commons-logging/commons-logging@1.1.1",
	"/packages/commons-vfs-1.0.jar":                                                                                                   "pkg:maven/commons-vfs/commons-vfs@1.0",
	"/packages/cxf-rt-transports-http-2.7.3.jar":                                                                                      "pkg:maven/org.apache.cxf/cxf-rt-transports-http@2.7.3",
	"/packages/dubbo-3.1.4.jar":                                                                                                       "pkg:maven/org.apache.dubbo/dubbo@3.1.4",
	"/packages/dubbo-3.1.4.jar:com.alibaba:hessian-lite":                                                                              "pkg:maven/com.alibaba/hessian-lite@3.2.13",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-auth":                                                                           "pkg:maven/org.apache.dubbo/dubbo-auth@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-cluster":                                                                        "pkg:maven/org.apache.dubbo/dubbo-cluster@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-common":                                                                         "pkg:maven/org.apache.dubbo/dubbo-common@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-compatible":                                                                     "pkg:maven/org.apache.dubbo/dubbo-compatible@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-config-api":                                                                     "pkg:maven/org.apache.dubbo/dubbo-config-api@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-config-spring":                                                                  "pkg:maven/org.apache.dubbo/dubbo-config-spring@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-configcenter-apollo":                                                            "pkg:maven/org.apache.dubbo/dubbo-configcenter-apollo@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-configcenter-nacos":                                                             "pkg:maven/org.apache.dubbo/dubbo-configcenter-nacos@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-configcenter-zookeeper":                                                         "pkg:maven/org.apache.dubbo/dubbo-configcenter-zookeeper@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-container-api":                                                                  "pkg:maven/org.apache.dubbo/dubbo-container-api@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-container-spring":                                                               "pkg:maven/org.apache.dubbo/dubbo-container-spring@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-filter-cache":                                                                   "pkg:maven/org.apache.dubbo/dubbo-filter-cache@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-filter-validation":                                                              "pkg:maven/org.apache.dubbo/dubbo-filter-validation@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-kubernetes":                                                                     "pkg:maven/org.apache.dubbo/dubbo-kubernetes@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-metadata-api":                                                                   "pkg:maven/org.apache.dubbo/dubbo-metadata-api@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-metadata-definition-protobuf":                                                   "pkg:maven/org.apache.dubbo/dubbo-metadata-definition-protobuf@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-metadata-report-nacos":                                                          "pkg:maven/org.apache.dubbo/dubbo-metadata-report-nacos@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-metadata-report-redis":                                                          "pkg:maven/org.apache.dubbo/dubbo-metadata-report-redis@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-metadata-report-zookeeper":                                                      "pkg:maven/org.apache.dubbo/dubbo-metadata-report-zookeeper@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-metrics-api":                                                                    "pkg:maven/org.apache.dubbo/dubbo-metrics-api@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-metrics-prometheus":                                                             "pkg:maven/org.apache.dubbo/dubbo-metrics-prometheus@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-monitor-api":                                                                    "pkg:maven/org.apache.dubbo/dubbo-monitor-api@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-monitor-default":                                                                "pkg:maven/org.apache.dubbo/dubbo-monitor-default@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-qos":                                                                            "pkg:maven/org.apache.dubbo/dubbo-qos@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-reactive":                                                                       "pkg:maven/org.apache.dubbo/dubbo-reactive@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-registry-api":                                                                   "pkg:maven/org.apache.dubbo/dubbo-registry-api@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-registry-multicast":                                                             "pkg:maven/org.apache.dubbo/dubbo-registry-multicast@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-registry-multiple":                                                              "pkg:maven/org.apache.dubbo/dubbo-registry-multiple@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-registry-nacos":                                                                 "pkg:maven/org.apache.dubbo/dubbo-registry-nacos@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-registry-zookeeper":                                                             "pkg:maven/org.apache.dubbo/dubbo-registry-zookeeper@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-remoting-api":                                                                   "pkg:maven/org.apache.dubbo/dubbo-remoting-api@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-remoting-http":                                                                  "pkg:maven/org.apache.dubbo/dubbo-remoting-http@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-remoting-netty":                                                                 "pkg:maven/org.apache.dubbo/dubbo-remoting-netty@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-remoting-netty4":                                                                "pkg:maven/org.apache.dubbo/dubbo-remoting-netty4@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-remoting-zookeeper":                                                             "pkg:maven/org.apache.dubbo/dubbo-remoting-zookeeper@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-remoting-zookeeper-curator5":                                                    "pkg:maven/org.apache.dubbo/dubbo-remoting-zookeeper-curator5@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-rpc-api":                                                                        "pkg:maven/org.apache.dubbo/dubbo-rpc-api@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-rpc-dubbo":                                                                      "pkg:maven/org.apache.dubbo/dubbo-rpc-dubbo@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-rpc-grpc":                                                                       "pkg:maven/org.apache.dubbo/dubbo-rpc-grpc@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-rpc-injvm":                                                                      "pkg:maven/org.apache.dubbo/dubbo-rpc-injvm@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-rpc-rest":                                                                       "pkg:maven/org.apache.dubbo/dubbo-rpc-rest@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-rpc-triple":                                                                     "pkg:maven/org.apache.dubbo/dubbo-rpc-triple@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-serialization-api":                                                              "pkg:maven/org.apache.dubbo/dubbo-serialization-api@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-serialization-fastjson2":                                                        "pkg:maven/org.apache.dubbo/dubbo-serialization-fastjson2@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-serialization-hessian2":                                                         "pkg:maven/org.apache.dubbo/dubbo-serialization-hessian2@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-serialization-jdk":                                                              "pkg:maven/org.apache.dubbo/dubbo-serialization-jdk@3.1.4",
	"/packages/dubbo-3.1.4.jar:org.apache.dubbo:dubbo-xds":                                                                            "pkg:maven/org.apache.dubbo/dubbo-xds@3.1.4",
	"/packages/elasticsearch-8.10.2.jar":                                                                                              "pkg:maven/org.elasticsearch/elasticsearch@8.10.2",
	"/packages/elasticsearch-rest-client-sniffer-7.17.11.jar":                                                                         "pkg:maven/org.elasticsearch.client/elasticsearch-rest-client-sniffer@7.17.11",
	"/packages/example-java-app-gradle-0.1.0.ear":                                                                                     "pkg:maven/example-java-app-gradle/example-java-app-gradle@0.1.0",
	"/packages/generic-webhook-trigger.hpi":                                                                                           "pkg:maven/org.jenkins-ci.plugins/generic-webhook-trigger@1.71",
	"/packages/generic-webhook-trigger.hpi:WEB-INF/lib/accessors-smart-1.2.jar":                                                       "pkg:maven/net.minidev/accessors-smart@1.2",
	"/packages/generic-webhook-trigger.hpi:WEB-INF/lib/asm-5.0.4.jar":                                                                 "pkg:maven/org.objectweb.asm/asm@5.0.4",
	"/packages/generic-webhook-trigger.hpi:WEB-INF/lib/commons-ip-math-1.32.jar":                                                      "pkg:maven/com.github.jgonian/commons-ip-math@1.32",
	"/packages/generic-webhook-trigger.hpi:WEB-INF/lib/generic-webhook-trigger.jar":                                                   "pkg:maven/org.jenkins-ci.plugins/generic-webhook-trigger@1.71",
	"/packages/generic-webhook-trigger.hpi:WEB-INF/lib/gson-2.8.2.jar":                                                                "pkg:maven/com.google.code.gson/gson@2.8.2",
	"/packages/generic-webhook-trigger.hpi:WEB-INF/lib/json-path-2.3.0.jar":                                                           "pkg:maven/com.jayway.jsonpath.json-path/json-path@2.3.0",
	"/packages/generic-webhook-trigger.hpi:WEB-INF/lib/json-smart-2.3.jar":                                                            "pkg:maven/net.minidev/json-smart@2.3",
	"/packages/geode-core-1.14.3.jar":                                                                                                 "pkg:maven/org.apache.geode/geode-core@1.14.3",
	"/packages/github-api-1.118.jar":                                                                                                  "pkg:maven/org.kohsuke/github-api@1.118",
	"/packages/google-oauth-client-1.25.0.jar":                                                                                        "pkg:maven/com.google.oauth-client/google-oauth-client@1.25.0",
	"/packages/gradle-7.1.1/lib/annotations-20.1.0.jar":                                                                               "pkg:maven/org.jetbrains.annotations/annotations@20.1.0",
	"/packages/gradle-7.1.1/lib/ant-1.10.9.jar":                                                                                       "pkg:maven/org.apache.ant/ant@1.10.9",
	"/packages/gradle-7.1.1/lib/ant-antlr-1.10.9.jar":                                                                                 "pkg:maven/org.apache.ant/ant-antlr@1.10.9",
	"/packages/gradle-7.1.1/lib/ant-junit-1.10.9.jar":                                                                                 "pkg:maven/org.apache.ant/ant-junit@1.10.9",
	"/packages/gradle-7.1.1/lib/ant-launcher-1.10.9.jar":                                                                              "pkg:maven/org.apache.ant/ant-launcher@1.10.9",
	"/packages/gradle-7.1.1/lib/antlr4-runtime-4.7.2.jar":                                                                             "pkg:maven/org.antlr/antlr4-runtime@4.7.2",
	"/packages/gradle-7.1.1/lib/asm-9.1.jar":                                                                                          "pkg:maven/org.objectweb.asm/asm@9.1",
	"/packages/gradle-7.1.1/lib/commons-compress-1.20.jar":                                                                            "pkg:maven/org.apache.commons/commons-compress@1.20",
	"/packages/gradle-7.1.1/lib/commons-io-2.6.jar":                                                                                   "pkg:maven/commons-io/commons-io@2.6",
	"/packages/gradle-7.1.1/lib/commons-lang-2.6.jar":                                                                                 "pkg:maven/commons-lang/commons-lang@2.6",
	"/packages/gradle-7.1.1/lib/failureaccess-1.0.1.jar":                                                                              "pkg:maven/com.google.guava/failureaccess@1.0.1",
	"/packages/gradle-7.1.1/lib/groovy-3.0.7.jar":                                                                                     "pkg:maven/org.codehaus.groovy/groovy@3.0.7",
	"/packages/gradle-7.1.1/lib/groovy-ant-3.0.7.jar":                                                                                 "pkg:maven/org.codehaus.groovy/groovy-ant@3.0.7",
	"/packages/gradle-7.1.1/lib/groovy-astbuilder-3.0.7.jar":                                                                          "pkg:maven/org.codehaus.groovy/groovy-astbuilder@3.0.7",
	"/packages/gradle-7.1.1/lib/groovy-console-3.0.7.jar":                                                                             "pkg:maven/org.codehaus.groovy/groovy-console@3.0.7",
	"/packages/gradle-7.1.1/lib/groovy-datetime-3.0.7.jar":                                                                            "pkg:maven/org.codehaus.groovy/groovy-datetime@3.0.7",
	"/packages/gradle-7.1.1/lib/groovy-dateutil-3.0.7.jar":                                                                            "pkg:maven/org.codehaus.groovy/groovy-dateutil@3.0.7",
	"/packages/gradle-7.1.1/lib/groovy-docgenerator-3.0.7.jar":                                                                        "pkg:maven/org.codehaus.groovy/groovy-docgenerator@3.0.7",
	"/packages/gradle-7.1.1/lib/groovy-groovydoc-3.0.7.jar":                                                                           "pkg:maven/org.codehaus.groovy/groovy-groovydoc@3.0.7",
	"/packages/gradle-7.1.1/lib/groovy-json-3.0.7.jar":                                                                                "pkg:maven/org.codehaus.groovy/groovy-json@3.0.7",
	"/packages/gradle-7.1.1/lib/groovy-nio-3.0.7.jar":                                                                                 "pkg:maven/org.codehaus.groovy/groovy-nio@3.0.7",
	"/packages/gradle-7.1.1/lib/groovy-sql-3.0.7.jar":                                                                                 "pkg:maven/org.codehaus.groovy/groovy-sql@3.0.7",
	"/packages/gradle-7.1.1/lib/groovy-swing-3.0.7.jar":                                                                               "pkg:maven/org.codehaus.groovy/groovy-swing@3.0.7",
	"/packages/gradle-7.1.1/lib/groovy-templates-3.0.7.jar":                                                                           "pkg:maven/org.codehaus.groovy/groovy-templates@3.0.7",
	"/packages/gradle-7.1.1/lib/groovy-test-3.0.7.jar":                                                                                "pkg:maven/org.codehaus.groovy/groovy-test@3.0.7",
	"/packages/gradle-7.1.1/lib/groovy-xml-3.0.7.jar":                                                                                 "pkg:maven/org.codehaus.groovy/groovy-xml@3.0.7",
	"/packages/gradle-7.1.1/lib/guava-27.1-android.jar":                                                                               "pkg:maven/com.google.guava/guava@27.1-android",
	"/packages/gradle-7.1.1/lib/jansi-1.18.jar":                                                                                       "pkg:maven/org.fusesource.jansi/jansi@1.18",
	"/packages/gradle-7.1.1/lib/jansi-1.18.jar:org.fusesource.hawtjni:hawtjni-runtime":                                                "pkg:maven/org.fusesource.hawtjni/hawtjni-runtime@1.17",
	"/packages/gradle-7.1.1/lib/jansi-1.18.jar:org.fusesource.jansi:jansi-freebsd32":                                                  "pkg:maven/org.fusesource.jansi/jansi-freebsd32@1.8",
	"/packages/gradle-7.1.1/lib/jansi-1.18.jar:org.fusesource.jansi:jansi-freebsd64":                                                  "pkg:maven/org.fusesource.jansi/jansi-freebsd64@1.8",
	"/packages/gradle-7.1.1/lib/jansi-1.18.jar:org.fusesource.jansi:jansi-linux32":                                                    "pkg:maven/org.fusesource.jansi/jansi-linux32@1.8",
	"/packages/gradle-7.1.1/lib/jansi-1.18.jar:org.fusesource.jansi:jansi-linux64":                                                    "pkg:maven/org.fusesource.jansi/jansi-linux64@1.8",
	"/packages/gradle-7.1.1/lib/jansi-1.18.jar:org.fusesource.jansi:jansi-native":                                                     "pkg:maven/org.fusesource.jansi/jansi-native@1.8",
	"/packages/gradle-7.1.1/lib/jansi-1.18.jar:org.fusesource.jansi:jansi-osx":                                                        "pkg:maven/org.fusesource.jansi/jansi-osx@1.8",
	"/packages/gradle-7.1.1/lib/jansi-1.18.jar:org.fusesource.jansi:jansi-windows32":                                                  "pkg:maven/org.fusesource.jansi/jansi-windows32@1.8",
	"/packages/gradle-7.1.1/lib/jansi-1.18.jar:org.fusesource.jansi:jansi-windows64":                                                  "pkg:maven/org.fusesource.jansi/jansi-windows64@1.8",
	"/packages/gradle-7.1.1/lib/javaparser-core-3.17.0.jar":                                                                           "pkg:maven/com.github.javaparser/javaparser-core@3.17.0",
	"/packages/gradle-7.1.1/lib/jcl-over-slf4j-1.7.30.jar":                                                                            "pkg:maven/org.slf4j/jcl-over-slf4j@1.7.30",
	"/packages/gradle-7.1.1/lib/jsr305-3.0.2.jar":                                                                                     "pkg:maven/com.google.code.findbugs/jsr305@3.0.2",
	"/packages/gradle-7.1.1/lib/jul-to-slf4j-1.7.30.jar":                                                                              "pkg:maven/org.slf4j/jul-to-slf4j@1.7.30",
	"/packages/gradle-7.1.1/lib/junit-4.13.2.jar":                                                                                     "pkg:maven/junit/junit@4.13.2",
	"/packages/gradle-7.1.1/lib/kryo-2.24.0.jar":                                                                                      "pkg:maven/com.esotericsoftware.kryo/kryo@2.24.0",
	"/packages/gradle-7.1.1/lib/kryo-2.24.0.jar:com.esotericsoftware.reflectasm:reflectasm":                                           "pkg:maven/com.esotericsoftware.reflectasm/reflectasm@1.09",
	"/packages/gradle-7.1.1/lib/log4j-over-slf4j-1.7.30.jar":                                                                          "pkg:maven/org.slf4j/log4j-over-slf4j@1.7.30",
	"/packages/gradle-7.1.1/lib/minlog-1.2.jar":                                                                                       "pkg:maven/com.esotericsoftware.minlog/minlog@1.2",
	"/packages/gradle-7.1.1/lib/objenesis-2.6.jar":                                                                                    "pkg:maven/org.objenesis/objenesis@2.6",
	"/packages/gradle-7.1.1/lib/plugins/aws-java-sdk-core-1.11.948.jar":                                                               "pkg:maven/com.amazonaws/aws-java-sdk-core@1.11.948",
	"/packages/gradle-7.1.1/lib/plugins/aws-java-sdk-kms-1.11.948.jar":                                                                "pkg:maven/com.amazonaws/aws-java-sdk-kms@1.11.948",
	"/packages/gradle-7.1.1/lib/plugins/aws-java-sdk-s3-1.11.948.jar":                                                                 "pkg:maven/com.amazonaws/aws-java-sdk-s3@1.11.948",
	"/packages/gradle-7.1.1/lib/plugins/aws-java-sdk-sts-1.11.948.jar":                                                                "pkg:maven/com.amazonaws/aws-java-sdk-sts@1.11.948",
	"/packages/gradle-7.1.1/lib/plugins/bcpg-jdk15on-1.68.jar":                                                                        "pkg:maven/org.bouncycastle/bcpg-jdk15on@1.68",
	"/packages/gradle-7.1.1/lib/plugins/bcpkix-jdk15on-1.68.jar":                                                                      "pkg:maven/org.bouncycastle/bcpkix-jdk15on@1.68",
	"/packages/gradle-7.1.1/lib/plugins/bcprov-jdk15on-1.68.jar":                                                                      "pkg:maven/org.bouncycastle/bcprov-jdk15on@1.68",
	"/packages/gradle-7.1.1/lib/plugins/commons-codec-1.15.jar":                                                                       "pkg:maven/commons-codec/commons-codec@1.15",
	"/packages/gradle-7.1.1/lib/plugins/dd-plist-1.21.jar":                                                                            "pkg:maven/com.googlecode.plist/dd-plist@1.21",
	"/packages/gradle-7.1.1/lib/plugins/google-api-services-storage-v1-rev171-1.25.0.jar:com.google.apis:google-api-services-storage": "pkg:maven/com.google.apis/google-api-services-storage@v1-rev171-1.25.0",
	"/packages/gradle-7.1.1/lib/plugins/google-http-client-1.25.0.jar":                                                                "pkg:maven/com.google.http-client/google-http-client@1.25.0",
	"/packages/gradle-7.1.1/lib/plugins/google-http-client-jackson2-1.25.0.jar":                                                       "pkg:maven/com.google.http-client/google-http-client-jackson2@1.25.0",
	"/packages/gradle-7.1.1/lib/plugins/google-oauth-client-1.25.0.jar":                                                               "pkg:maven/com.google.oauth-client/google-oauth-client@1.25.0",
	"/packages/gradle-7.1.1/lib/plugins/gson-2.8.6.jar":                                                                               "pkg:maven/com.google.code.gson/gson@2.8.6",
	"/packages/gradle-7.1.1/lib/plugins/httpclient-4.5.13.jar":                                                                        "pkg:maven/org.apache.httpcomponents/httpclient@4.5.13",
	"/packages/gradle-7.1.1/lib/plugins/httpcore-4.4.14.jar":                                                                          "pkg:maven/org.apache.httpcomponents/httpcore@4.4.14",
	"/packages/gradle-7.1.1/lib/plugins/ion-java-1.0.2.jar":                                                                           "pkg:maven/software.amazon.ion/ion-java@1.0.2",
	"/packages/gradle-7.1.1/lib/plugins/jackson-annotations-2.12.1.jar":                                                               "pkg:maven/com.fasterxml.jackson.core/jackson-annotations@2.12.1",
	"/packages/gradle-7.1.1/lib/plugins/jackson-core-2.12.1.jar":                                                                      "pkg:maven/com.fasterxml.jackson.core/jackson-core@2.12.1",
	"/packages/gradle-7.1.1/lib/plugins/jackson-databind-2.12.1.jar":                                                                  "pkg:maven/com.fasterxml.jackson.core/jackson-databind@2.12.1",
	"/packages/gradle-7.1.1/lib/plugins/jakarta.activation-2.0.0.jar":                                                                 "pkg:maven/com.sun.activation/jakarta.activation@2.0.0",
	"/packages/gradle-7.1.1/lib/plugins/jakarta.xml.bind-api-3.0.0.jar":                                                               "pkg:maven/jakarta.xml.bind/jakarta.xml.bind-api@3.0.0",
	"/packages/gradle-7.1.1/lib/plugins/jatl-0.2.3.jar":                                                                               "pkg:maven/com.googlecode.jatl/jatl@0.2.3",
	"/packages/gradle-7.1.1/lib/plugins/jaxb-core-3.0.0.jar:com.sun.istack:istack-commons-runtime":                                    "pkg:maven/com.sun.istack/istack-commons-runtime@4.0.0",
	"/packages/gradle-7.1.1/lib/plugins/jaxb-core-3.0.0.jar:org.glassfish.jaxb:txw2":                                                  "pkg:maven/org.glassfish.jaxb/txw2@3.0.0",
	"/packages/gradle-7.1.1/lib/plugins/jaxb-impl-3.0.0.jar":                                                                          "pkg:maven/com.sun.xml.bind/jaxb-impl@3.0.0",
	"/packages/gradle-7.1.1/lib/plugins/jaxb-impl-3.0.0.jar:org.glassfish.jaxb:jaxb-runtime":                                          "pkg:maven/org.glassfish.jaxb/jaxb-runtime@3.0.0",
	"/packages/gradle-7.1.1/lib/plugins/jcifs-1.3.17.jar":                                                                             "pkg:maven/jcifs/jcifs@1.3.17",
	"/packages/gradle-7.1.1/lib/plugins/jmespath-java-1.11.948.jar":                                                                   "pkg:maven/com.amazonaws/jmespath-java@1.11.948",
	"/packages/gradle-7.1.1/lib/plugins/joda-time-2.10.4.jar":                                                                         "pkg:maven/joda-time/joda-time@2.10.4",
	"/packages/gradle-7.1.1/lib/plugins/jsch-0.1.55.jar":                                                                              "pkg:maven/com.jcraft/jsch@0.1.55",
	"/packages/gradle-7.1.1/lib/plugins/jzlib-1.1.3.jar":                                                                              "pkg:maven/com.jcraft/jzlib@1.1.3",
	"/packages/gradle-7.1.1/lib/plugins/maven-builder-support-3.6.3.jar":                                                              "pkg:maven/org.apache.maven/maven-builder-support@3.6.3",
	"/packages/gradle-7.1.1/lib/plugins/maven-model-3.6.3.jar":                                                                        "pkg:maven/org.apache.maven/maven-model@3.6.3",
	"/packages/gradle-7.1.1/lib/plugins/maven-repository-metadata-3.6.3.jar":                                                          "pkg:maven/org.apache.maven/maven-repository-metadata@3.6.3",
	"/packages/gradle-7.1.1/lib/plugins/maven-settings-3.6.3.jar":                                                                     "pkg:maven/org.apache.maven/maven-settings@3.6.3",
	"/packages/gradle-7.1.1/lib/plugins/maven-settings-builder-3.6.3.jar":                                                             "pkg:maven/org.apache.maven/maven-settings-builder@3.6.3",
	"/packages/gradle-7.1.1/lib/plugins/opentest4j-1.2.0.jar":                                                                         "pkg:maven/org.opentest4j/opentest4j@1.2.0",
	"/packages/gradle-7.1.1/lib/plugins/org.eclipse.jgit-5.7.0.202003110725-r.jar":                                                    "pkg:maven/org.eclipse.jgit/org.eclipse.jgit@5.7.0.202003110725-r",
	"/packages/gradle-7.1.1/lib/plugins/plexus-cipher-1.7.jar":                                                                        "pkg:maven/org.sonatype.plexus/plexus-cipher@1.7",
	"/packages/gradle-7.1.1/lib/plugins/plexus-interpolation-1.26.jar":                                                                "pkg:maven/org.codehaus.plexus/plexus-interpolation@1.26",
	"/packages/gradle-7.1.1/lib/plugins/plexus-sec-dispatcher-1.4.jar":                                                                "pkg:maven/org.sonatype.plexus/plexus-sec-dispatcher@1.4",
	"/packages/gradle-7.1.1/lib/plugins/plexus-utils-3.3.0.jar":                                                                       "pkg:maven/org.codehaus.plexus/plexus-utils@3.3.0",
	"/packages/gradle-7.1.1/lib/plugins/snakeyaml-1.28.jar":                                                                           "pkg:maven/org.yaml/snakeyaml@1.28",
	"/packages/gradle-7.1.1/lib/plugins/testng-6.3.1.jar":                                                                             "pkg:maven/org.testng/testng@6.3.1",
	"/packages/gradle-7.1.1/lib/plugins/xercesImpl-2.12.0.jar":                                                                        "pkg:maven/org.apache.xerces.impl.Version/xercesImpl@2.12.0",
	"/packages/gradle-7.1.1/lib/qdox-1.12.1.jar":                                                                                      "pkg:maven/com.thoughtworks.qdox/qdox@1.12.1",
	"/packages/gradle-7.1.1/lib/slf4j-api-1.7.30.jar":                                                                                 "pkg:maven/org.slf4j/slf4j-api@1.7.30",
	"/packages/gradle.hpi": "pkg:maven/org.jenkins-ci.plugins/gradle@1.36",
	"/packages/gradle.hpi:WEB-INF/lib/gradle-plugin-1.36.jar":                              "pkg:maven/org.jenkins-ci.plugins/gradle-plugin@1.36",
	"/packages/graphql-java-20.0.jar":                                                      "pkg:maven/com.graphql-java/graphql-java@20.0",
	"/packages/graphql-java-20.0.jar:com.google.guava:guava":                               "pkg:maven/com.google.guava/guava@31.0.1-jre",
	"/packages/graphql-java-20.0.jar:org.antlr:antlr4-runtime":                             "pkg:maven/org.antlr/antlr4-runtime@4.9.3",
	"/packages/guava-29.0-jre.jar":                                                         "pkg:maven/com.google.guava/guava@29.0-jre",
	"/packages/hadoop-common-3.3.2.jar":                                                    "pkg:maven/org.apache.hadoop/hadoop-common@3.3.2",
	"/packages/hazelcast-5.2.4.jar":                                                        "pkg:maven/com.hazelcast/hazelcast@5.2.4",
	"/packages/hazelcast-5.2.4.jar:com.fasterxml.jackson.core:jackson-core":                "pkg:maven/com.fasterxml.jackson.core/jackson-core@2.15.2",
	"/packages/hazelcast-5.2.4.jar:com.fasterxml.jackson.jr:jackson-jr-annotation-support": "pkg:maven/com.fasterxml.jackson.jr/jackson-jr-annotation-support@2.15.2",
	"/packages/hazelcast-5.2.4.jar:com.fasterxml.jackson.jr:jackson-jr-objects":            "pkg:maven/com.fasterxml.jackson.jr/jackson-jr-objects@2.15.2",
	"/packages/hazelcast-5.2.4.jar:com.github.erosb:everit-json-schema":                    "pkg:maven/com.github.erosb/everit-json-schema@1.14.2",
	"/packages/hazelcast-5.2.4.jar:com.zaxxer:HikariCP":                                    "pkg:maven/com.zaxxer/HikariCP@4.0.3",
	"/packages/hazelcast-5.2.4.jar:io.github.classgraph:classgraph":                        "pkg:maven/io.github.classgraph/classgraph@4.8.149",
	"/packages/hazelcast-5.2.4.jar:org.json:json":                                          "pkg:maven/org.json/json@20230227",
	"/packages/hazelcast-5.2.4.jar:org.snakeyaml:snakeyaml-engine":                         "pkg:maven/org.snakeyaml/snakeyaml-engine@2.3",
	"/packages/hibernate-validator-6.1.3.Final.jar":                                        "pkg:maven/org.hibernate.validator/hibernate-validator@6.1.3.Final",
	"/packages/http2-common-11.0.16.jar":                                                   "pkg:maven/org.eclipse.jetty.http2/http2-common@11.0.16",
	"/packages/hudson.war:WEB-INF/hudson-cli.jar:args4j:args4j":                            "pkg:maven/args4j/args4j@2.0.16",
	"/packages/hudson.war:WEB-INF/hudson-cli.jar:org.jvnet:animal-sniffer-annotation":      "pkg:maven/org.jvnet/animal-sniffer-annotation@1.0",
	"/packages/hudson.war:WEB-INF/hudson-cli.jar:org.jvnet.hudson.main:remoting":           "pkg:maven/org.jvnet.hudson.main/remoting@1.390",
	"/packages/hudson.war:WEB-INF/hudson-cli.jar:org.jvnet.localizer:localizer":            "pkg:maven/org.jvnet.localizer/localizer@1.10",
	// "/packages/hudson.war:WEB-INF/hudson-cli.jar":                                          "pkg:maven/org.jvnet.hudson.main/hudson-cli@1.390",
	"/packages/hudson.war:WEB-INF/lib/access-modifier-annotation-1.0.jar": "pkg:maven/org.kohsuke/access-modifier-annotation@1.0",
	"/packages/hudson.war:WEB-INF/lib/acegi-security-1.0.5.jar":           "pkg:maven/org.acegisecurity/acegi-security@1.0.5",
	"/packages/hudson.war:WEB-INF/lib/activation-1.1.1-hudson-1.jar":      "pkg:maven/org.jvnet.hudson/activation@1.1.1-hudson-1",
	"/packages/hudson.war:WEB-INF/lib/akuma-1.2.jar":                      "pkg:maven/com.sun.akuma/akuma@1.2",
	"/packages/hudson.war:WEB-INF/lib/annotation-indexer-1.2.jar":         "pkg:maven/org.jvnet.hudson/annotation-indexer@1.2",
	"/packages/hudson.war:WEB-INF/lib/ant-1.8.0.jar":                      "pkg:maven/org.apache.ant/ant@1.8.0",
	"/packages/hudson.war:WEB-INF/lib/ant-launcher-1.8.0.jar":             "pkg:maven/org.apache.ant/ant-launcher@1.8.0",
	"/packages/hudson.war:WEB-INF/lib/antlr-2.7.6.jar":                    "pkg:maven/antlr/antlr@2.7.6",
	"/packages/hudson.war:WEB-INF/lib/aopalliance-1.0.jar":                "pkg:maven/aopalliance/aopalliance@1.0",
	"/packages/hudson.war:WEB-INF/lib/args4j-2.0.16.jar":                  "pkg:maven/args4j/args4j@2.0.16",
	// "/packages/hudson.war:WEB-INF/lib/asm-2.2.3.jar":                                                                                       "pkg:maven/asm/asm@2.2.3",
	// "/packages/hudson.war:WEB-INF/lib/asm-commons-2.2.3.jar":                                                                               "pkg:maven/asm/asm-commons@2.2.3",
	// "/packages/hudson.war:WEB-INF/lib/asm-tree-2.2.3.jar":                                                                                  "pkg:maven/asm/asm-tree@2.2.3",
	"/packages/hudson.war:WEB-INF/lib/avalon-framework-4.1.3.jar":                          "pkg:maven/avalon-framework/avalon-framework@4.1.3",
	"/packages/hudson.war:WEB-INF/lib/bridge-method-annotation-1.2.jar":                    "pkg:maven/com.infradna.tool/bridge-method-annotation@1.2",
	"/packages/hudson.war:WEB-INF/lib/cli-1.390.jar":                                       "pkg:maven/org.jvnet.hudson.main/cli@1.390",
	"/packages/hudson.war:WEB-INF/lib/commons-beanutils-1.8.0.jar":                         "pkg:maven/commons-beanutils/commons-beanutils@1.8.0",
	"/packages/hudson.war:WEB-INF/lib/commons-codec-1.4.jar":                               "pkg:maven/commons-codec/commons-codec@1.4",
	"/packages/hudson.war:WEB-INF/lib/commons-collections-3.2.jar":                         "pkg:maven/commons-collections/commons-collections@3.2",
	"/packages/hudson.war:WEB-INF/lib/commons-digester-1.7.jar":                            "pkg:maven/commons-digester/commons-digester@1.7",
	"/packages/hudson.war:WEB-INF/lib/commons-discovery-0.4.jar":                           "pkg:maven/commons-discovery/commons-discovery@0.4",
	"/packages/hudson.war:WEB-INF/lib/commons-fileupload-1.2.1.jar":                        "pkg:maven/commons-fileupload/commons-fileupload@1.2.1",
	"/packages/hudson.war:WEB-INF/lib/commons-io-1.4.jar":                                  "pkg:maven/commons-io/commons-io@1.4",
	"/packages/hudson.war:WEB-INF/lib/commons-jelly-1.1-hudson-20100305.jar":               "pkg:maven/org.jvnet.hudson/commons-jelly@1.1-hudson-20100305",
	"/packages/hudson.war:WEB-INF/lib/commons-jelly-tags-define-1.0.1-hudson-20071021.jar": "pkg:maven/org.jvnet.hudson/commons-jelly-tags-define@1.0.1-hudson-20071021",
	"/packages/hudson.war:WEB-INF/lib/commons-jelly-tags-fmt-1.0.jar":                      "pkg:maven/commons-jelly/commons-jelly-tags-fmt@1.0",
	"/packages/hudson.war:WEB-INF/lib/commons-jelly-tags-xml-1.1.jar":                      "pkg:maven/commons-jelly/commons-jelly-tags-xml@1.1",
	"/packages/hudson.war:WEB-INF/lib/commons-jexl-1.1-hudson-20090508.jar":                "pkg:maven/org.jvnet.hudson/commons-jexl@1.1-hudson-20090508",
	"/packages/hudson.war:WEB-INF/lib/commons-lang-2.4.jar":                                "pkg:maven/commons-lang/commons-lang@2.4",
	"/packages/hudson.war:WEB-INF/lib/commons-logging-1.1.jar":                             "pkg:maven/commons-logging/commons-logging@1.1",
	"/packages/hudson.war:WEB-INF/lib/commons-pool-1.3.jar":                                "pkg:maven/commons-pool/commons-pool@1.3",
	"/packages/hudson.war:WEB-INF/lib/crypto-util-1.0.jar":                                 "pkg:maven/org.jvnet.hudson/crypto-util@1.0",
	// "/packages/hudson.war:WEB-INF/lib/dom4j-1.6.1-hudson-3.jar":             "pkg:maven/org.jvnet.hudson.dom4j/dom4j@1.6.1-hudson-3",
	"/packages/hudson.war:WEB-INF/lib/embedded_su4j-1.1.jar":              "pkg:maven/com.sun.solaris/embedded_su4j@1.1",
	"/packages/hudson.war:WEB-INF/lib/ezmorph-1.0.3.jar":                  "pkg:maven/net.sf.ezmorph/ezmorph@1.0.3",
	"/packages/hudson.war:WEB-INF/lib/graph-layouter-1.0.jar":             "pkg:maven/org.kohsuke/graph-layouter@1.0",
	"/packages/hudson.war:WEB-INF/lib/groovy-all-1.6.0.jar":               "pkg:maven/org.codehaus.groovy/groovy-all@1.6.0",
	"/packages/hudson.war:WEB-INF/lib/guava-r06.jar":                      "pkg:maven/com.google.guava/guava@r06",
	"/packages/hudson.war:WEB-INF/lib/hudson-core-1.390.jar":              "pkg:maven/org.jvnet.hudson.main/hudson-core@1.390",
	"/packages/hudson.war:WEB-INF/lib/j-interop-2.0.5.jar":                "pkg:maven/j-interop/j-interop@2.0.5",
	"/packages/hudson.war:WEB-INF/lib/j-interopdeps-2.0.5.jar":            "pkg:maven/j-interopdeps/j-interopdeps@2.0.5",
	"/packages/hudson.war:WEB-INF/lib/jaxen-1.1-beta-11.jar":              "pkg:maven/org.jaxen/jaxen@1.1-beta-11",
	"/packages/hudson.war:WEB-INF/lib/jcaptcha-all-1.0-RC6.jar":           "pkg:maven/jcaptcha-all/jcaptcha-all@1.0-RC6",
	"/packages/hudson.war:WEB-INF/lib/jcifs-1.3.14-kohsuke-1.jar":         "pkg:maven/org.samba.jcifs/jcifs@1.3.14-kohsuke-1",
	"/packages/hudson.war:WEB-INF/lib/jcommon-1.0.12.jar":                 "pkg:maven/jfree/jcommon@1.0.12",
	"/packages/hudson.war:WEB-INF/lib/jfreechart-1.0.9.jar":               "pkg:maven/jfreechart/jfreechart@1.0.9",
	"/packages/hudson.war:WEB-INF/lib/jinterop-proxy-1.1.jar":             "pkg:maven/org.kohsuke.jinterop/jinterop-proxy@1.1",
	"/packages/hudson.war:WEB-INF/lib/jinterop-wmi-1.0.jar":               "pkg:maven/org.jvnet.hudson/jinterop-wmi@1.0",
	"/packages/hudson.war:WEB-INF/lib/jline-0.9.94.jar":                   "pkg:maven/jline/jline@0.9.94",
	"/packages/hudson.war:WEB-INF/lib/jmdns-3.1.6-hudson-2.jar":           "pkg:maven/com.strangeberry.jmdns.tools.Main/jmdns@3.1.6-hudson-2",
	"/packages/hudson.war:WEB-INF/lib/jna-3.2.4.jar":                      "pkg:maven/com.sun.jna/jna@3.2.4",
	"/packages/hudson.war:WEB-INF/lib/jna-posix-1.0.3.jar":                "pkg:maven/org.jruby.ext.posix/jna-posix@1.0.3",
	"/packages/hudson.war:WEB-INF/lib/json-lib-2.1-rev6.jar":              "pkg:maven/json-lib/json-lib@2.1-rev6",
	"/packages/hudson.war:WEB-INF/lib/jstl-1.1.0.jar":                     "pkg:maven/com.sun/jstl@1.1.0",
	"/packages/hudson.war:WEB-INF/lib/jtidy-4aug2000r7-dev-hudson-1.jar":  "pkg:maven/jtidy/jtidy@4aug2000r7-dev-hudson-1",
	"/packages/hudson.war:WEB-INF/lib/libpam4j-1.2.jar":                   "pkg:maven/org.jvnet.libpam4j/libpam4j@1.2",
	"/packages/hudson.war:WEB-INF/lib/libzfs-0.5.jar":                     "pkg:maven/org.jvnet.libzfs/libzfs@0.5",
	"/packages/hudson.war:WEB-INF/lib/localizer-1.10.jar":                 "pkg:maven/org.jvnet.localizer/localizer@1.10",
	"/packages/hudson.war:WEB-INF/lib/log4j-1.2.9.jar":                    "pkg:maven/log4j/log4j@1.2.9",
	"/packages/hudson.war:WEB-INF/lib/logkit-1.0.1.jar":                   "pkg:maven/logkit/logkit@1.0.1",
	"/packages/hudson.war:WEB-INF/lib/mail-1.4.jar":                       "pkg:maven/com.sun/mail@1.4",
	"/packages/hudson.war:WEB-INF/lib/memory-monitor-1.3.jar":             "pkg:maven/org.jvnet.hudson/memory-monitor@1.3",
	"/packages/hudson.war:WEB-INF/lib/oro-2.0.8.jar":                      "pkg:maven/org.apache.oro/oro@2.0.8",
	"/packages/hudson.war:WEB-INF/lib/remoting-1.390.jar:args4j:args4j":   "pkg:maven/args4j/args4j@2.0.16",
	"/packages/hudson.war:WEB-INF/lib/remoting-1.390.jar":                 "pkg:maven/org.jvnet.hudson.main/remoting@1.390",
	"/packages/hudson.war:WEB-INF/lib/robust-http-client-1.1.jar":         "pkg:maven/org.jvnet.robust-http-client/robust-http-client@1.1",
	"/packages/hudson.war:WEB-INF/lib/sezpoz-1.7.jar":                     "pkg:maven/net.java.sezpoz/sezpoz@1.7",
	"/packages/hudson.war:WEB-INF/lib/spring-aop-2.5.jar":                 "pkg:maven/org.springframework/spring-aop@2.5",
	"/packages/hudson.war:WEB-INF/lib/spring-beans-2.5.jar":               "pkg:maven/org.springframework/spring-beans@2.5",
	"/packages/hudson.war:WEB-INF/lib/spring-context-2.5.jar":             "pkg:maven/org.springframework/spring-context@2.5",
	"/packages/hudson.war:WEB-INF/lib/spring-core-2.5.jar":                "pkg:maven/org.springframework/spring-core@2.5",
	"/packages/hudson.war:WEB-INF/lib/spring-dao-1.2.9.jar":               "pkg:maven/org.springframework/spring-dao@1.2.9",
	"/packages/hudson.war:WEB-INF/lib/spring-jdbc-1.2.9.jar":              "pkg:maven/org.springframework/spring-jdbc@1.2.9",
	"/packages/hudson.war:WEB-INF/lib/spring-web-2.5.jar":                 "pkg:maven/org.springframework/spring-web@2.5",
	"/packages/hudson.war:WEB-INF/lib/stapler-1.155.jar":                  "pkg:maven/org.kohsuke.stapler/stapler@1.155",
	"/packages/hudson.war:WEB-INF/lib/stapler-adjunct-timeline-1.2.jar":   "pkg:maven/org.kohsuke.stapler/stapler-adjunct-timeline@1.2",
	"/packages/hudson.war:WEB-INF/lib/stapler-jelly-1.155.jar":            "pkg:maven/org.kohsuke.stapler/stapler-jelly@1.155",
	"/packages/hudson.war:WEB-INF/lib/stax-api-1.0.1.jar":                 "pkg:maven/stax-api/stax-api@1.0.1",
	"/packages/hudson.war:WEB-INF/lib/task-reactor-1.2.jar":               "pkg:maven/org.jvnet.hudson/task-reactor@1.2",
	"/packages/hudson.war:WEB-INF/lib/tiger-types-1.3.jar":                "pkg:maven/org.jvnet/tiger-types@1.3",
	"/packages/hudson.war:WEB-INF/lib/trilead-putty-extension-1.0.jar":    "pkg:maven/org.kohsuke/trilead-putty-extension@1.0",
	"/packages/hudson.war:WEB-INF/lib/trilead-ssh2-build212-hudson-5.jar": "pkg:maven/org.jvnet.hudson/trilead-ssh2@build212-hudson-5",
	"/packages/hudson.war:WEB-INF/lib/txw2-20070624.jar":                  "pkg:maven/txw2/txw2@20070624",
	"/packages/hudson.war:WEB-INF/lib/windows-remote-command-1.0.jar":     "pkg:maven/org.jvnet.hudson/windows-remote-command@1.0",
	"/packages/hudson.war:WEB-INF/lib/winp-1.14.jar":                      "pkg:maven/org.jvnet.winp/winp@1.14",
	"/packages/hudson.war:WEB-INF/lib/wstx-asl-3.2.7.jar":                 "pkg:maven/wstx-asl/wstx-asl@3.2.7",
	// "/packages/hudson.war:WEB-INF/lib/xpp3_min-1.1.4c.jar":                                                                "pkg:maven/xpp3_min/xpp3_min@1.1.4c",
	// "/packages/hudson.war:WEB-INF/lib/xpp3-1.1.4c.jar":                                                                    "pkg:maven/xpp3/xpp3@1.1.4c",
	"/packages/hudson.war:WEB-INF/lib/xstream-1.3.1-hudson-8.jar":                                                         "pkg:maven/org.jvnet.hudson/xstream@1.3.1-hudson-8",
	"/packages/hudson.war:WEB-INF/plugins/cvs.hpi":                                                                        "pkg:maven/org.jvnet.hudson.plugins/cvs@1.2",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/avalon-framework-4.1.3.jar":                        "pkg:maven/avalon-framework/avalon-framework@4.1.3",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/classworlds-1.1.jar":                               "pkg:maven/org.codehaus.classworlds/classworlds@1.1",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/commons-codec-1.4.jar":                             "pkg:maven/commons-codec/commons-codec@1.4",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/commons-httpclient-3.1-rc1.jar":                    "pkg:maven/commons-httpclient/commons-httpclient@3.1-rc1",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/commons-logging-1.1.jar":                           "pkg:maven/commons-logging/commons-logging@1.1",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/doxia-sink-api-1.0-alpha-10.jar":                   "pkg:maven/org.apache.maven.doxia/doxia-sink-api@1.0-alpha-10",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/jsch-0.1.27.jar":                                   "pkg:maven/jsch/jsch@0.1.27",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/log4j-1.2.9.jar":                                   "pkg:maven/log4j/log4j@1.2.9",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/logkit-1.0.1.jar":                                  "pkg:maven/logkit/logkit@1.0.1",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/maven-agent-1.390.jar":                             "pkg:maven/org.jvnet.hudson.main/maven-agent@1.390",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/maven-artifact-2.0.9.jar":                          "pkg:maven/org.apache.maven/maven-artifact@2.0.9",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/maven-artifact-manager-2.0.9.jar":                  "pkg:maven/org.apache.maven/maven-artifact-manager@2.0.9",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/maven-core-2.0.9.jar":                              "pkg:maven/org.apache.maven/maven-core@2.0.9",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/maven-embedder-2.0.4-hudson-1.jar":                 "pkg:maven/org.jvnet.hudson/maven-embedder@2.0.4-hudson-1",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/maven-embedder-2.0.4.jar":                          "pkg:maven/org.apache.maven/maven-embedder@2.0.4",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/maven-error-diagnostics-2.0.9.jar":                 "pkg:maven/org.apache.maven/maven-error-diagnostics@2.0.9",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/maven-interceptor-1.390.jar":                       "pkg:maven/org.jvnet.hudson.main/maven-interceptor@1.390",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/maven-model-2.0.9.jar":                             "pkg:maven/org.apache.maven/maven-model@2.0.9",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/maven-monitor-2.0.9.jar":                           "pkg:maven/org.apache.maven/maven-monitor@2.0.9",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/maven-plugin-api-2.0.9.jar":                        "pkg:maven/org.apache.maven/maven-plugin-api@2.0.9",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/maven-plugin-descriptor-2.0.9.jar":                 "pkg:maven/org.apache.maven/maven-plugin-descriptor@2.0.9",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/maven-plugin-parameter-documenter-2.0.9.jar":       "pkg:maven/org.apache.maven/maven-plugin-parameter-documenter@2.0.9",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/maven-plugin-registry-2.0.9.jar":                   "pkg:maven/org.apache.maven/maven-plugin-registry@2.0.9",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/maven-profile-2.0.9.jar":                           "pkg:maven/org.apache.maven/maven-profile@2.0.9",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/maven-project-2.0.9.jar":                           "pkg:maven/org.apache.maven/maven-project@2.0.9",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/maven-reporting-api-2.0.9.jar":                     "pkg:maven/org.apache.maven.reporting/maven-reporting-api@2.0.9",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/maven-repository-metadata-2.0.9.jar":               "pkg:maven/org.apache.maven/maven-repository-metadata@2.0.9",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/maven-settings-2.0.9.jar":                          "pkg:maven/org.apache.maven/maven-settings@2.0.9",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/maven2.1-interceptor-1.2.jar":                      "pkg:maven/org.jvnet.hudson/maven2.1-interceptor@1.2",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/plexus-container-default-1.0-alpha-9-stable-1.jar": "pkg:maven/org.codehaus.plexus/plexus-container-default@1.0-alpha-9-stable-1",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/plexus-interactivity-api-1.0-alpha-4.jar":          "pkg:maven/org.codehaus.plexus/plexus-interactivity-api@1.0-alpha-4",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/plexus-utils-1.5.1.jar":                            "pkg:maven/org.codehaus.plexus/plexus-utils@1.5.1",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/slide-webdavlib-2.1.jar":                           "pkg:maven/slide-webdavlib/slide-webdavlib@2.1",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/wagon-file-1.0-beta-2.jar":                         "pkg:maven/org.apache.maven.wagon/wagon-file@1.0-beta-2",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/wagon-http-lightweight-1.0-beta-2.jar":             "pkg:maven/org.apache.maven.wagon/wagon-http-lightweight@1.0-beta-2",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/wagon-http-shared-1.0-beta-2.jar":                  "pkg:maven/org.apache.maven.wagon/wagon-http-shared@1.0-beta-2",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/wagon-provider-api-1.0-beta-2.jar":                 "pkg:maven/org.apache.maven.wagon/wagon-provider-api@1.0-beta-2",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/wagon-ssh-1.0-beta-2.jar":                          "pkg:maven/org.apache.maven.wagon/wagon-ssh@1.0-beta-2",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/wagon-ssh-common-1.0-beta-2.jar":                   "pkg:maven/org.apache.maven.wagon/wagon-ssh-common@1.0-beta-2",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/wagon-ssh-external-1.0-beta-2.jar":                 "pkg:maven/org.apache.maven.wagon/wagon-ssh-external@1.0-beta-2",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/wagon-webdav-1.0-beta-2-hudson-1.jar":              "pkg:maven/org.jvnet.hudson/wagon-webdav@1.0-beta-2-hudson-1",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi:WEB-INF/lib/xml-im-exporter-1.1.jar":                           "pkg:maven/xml-im-exporter/xml-im-exporter@1.1",
	"/packages/hudson.war:WEB-INF/plugins/maven-plugin.hpi":                                                               "pkg:maven/org.jvnet.hudson.main/maven-plugin@1.390",
	"/packages/hudson.war:WEB-INF/plugins/ssh-slaves.hpi":                                                                 "pkg:maven/org.jvnet.hudson.plugins/ssh-slaves@0.14",
	"/packages/hudson.war:WEB-INF/plugins/subversion.hpi:WEB-INF/lib/svnkit-1.3.4-hudson-2.jar":                           "pkg:maven/svnkit/svnkit@1.3.4-hudson-2",
	"/packages/hudson.war:WEB-INF/plugins/subversion.hpi":                                                                 "pkg:maven/org.jvnet.hudson.plugins/subversion@1.20",
	"/packages/hudson.war:WEB-INF/remoting.jar:args4j:args4j":                                                             "pkg:maven/args4j/args4j@2.0.16",
	"/packages/hudson.war:WEB-INF/remoting.jar":                                                                           "pkg:maven/org.jvnet.hudson.main/remoting@1.390",
	"/packages/hudson.war:WEB-INF/slave.jar:args4j:args4j":                                                                "pkg:maven/args4j/args4j@2.0.16",
	"/packages/hudson.war:WEB-INF/slave.jar:org.jvnet.hudson.main:remoting":                                               "pkg:maven/org.jvnet.hudson.main/remoting@1.390",
	// "/packages/hudson.war:WEB-INF/slave.jar":                                                                                               "pkg:maven/org.jvnet.hudson.main/remoting@1.390",
	"/packages/hudson.war:winstone.jar":           "pkg:maven/org.jvnet.hudson.winstone/winstone@0.9.10-hudson-24",
	"/packages/hudson.war":                        "pkg:maven/org.jvnet.hudson.main/hudson-war@1.390",
	"/packages/infinispan-core-11.0.4.Final.jar":  "pkg:maven/org.infinispan/infinispan-core@11.0.4.Final",
	"/packages/jeecg-boot-common-3.6.0.jar":       "pkg:maven/org.jeecgframework.boot/jeecg-boot-common@3.6.0",
	"/packages/jetty-client-9.4.31.v20200723.jar": "pkg:maven/org.eclipse.jetty/jetty-client@9.4.31.v20200723",
	"/packages/jetty-server-12.0.0.alpha3.jar":    "pkg:maven/org.eclipse.jetty/jetty-server@12.0.0.alpha3",
	"/packages/jetty-setuid-java-1.0.4.jar":       "pkg:maven/org.eclipse.jetty.toolchain.setuid/jetty-setuid-java@1.0.4",
	"/packages/jmdns-3.4.1.jar":                   "pkg:maven/javax.jmdns/jmdns@3.4.1",
	"/packages/jruby-stdlib-9.1.15.0.jar":         "pkg:maven/org.jruby/jruby-stdlib@9.1.15.0",
	"/packages/jruby-stdlib-9.1.15.0.jar:META-INF/jruby.home/lib/ruby/stdlib/org/bouncycastle/bcpkix-jdk15on/1.56/bcpkix-jdk15on-1.56.jar": "pkg:maven/org.bouncycastle/bcpkix-jdk15on@1.56",
	"/packages/jruby-stdlib-9.1.15.0.jar:META-INF/jruby.home/lib/ruby/stdlib/org/bouncycastle/bcprov-jdk15on/1.56/bcprov-jdk15on-1.56.jar": "pkg:maven/org.bouncycastle/bcprov-jdk15on@1.56",
	"/packages/jruby-stdlib-9.1.15.0.jar:META-INF/jruby.home/lib/ruby/stdlib/org/yaml/snakeyaml/1.18/snakeyaml-1.18.jar":                   "pkg:maven/org.yaml/snakeyaml@1.18",
	"/packages/jsch-0.1.55.jar":                                    "pkg:maven/com.jcraft/jsch@0.1.55",
	"/packages/junit-4.13.1.jar":                                   "pkg:maven/junit/junit@4.13.1",
	"/packages/junit.hpi":                                          "pkg:maven/org.jenkins-ci.plugins/junit@1.25",
	"/packages/junit.hpi:WEB-INF/lib/junit.jar":                    "pkg:maven/org.jenkins-ci.plugins/junit@1.25",
	"/packages/kafka_2.13-3.2.2.jar":                               "pkg:maven/org.apache.kafka/kafka_2.13@3.2.2",
	"/packages/keycloak-core-22.0.2.jar":                           "pkg:maven/org.keycloak/keycloak-core@22.0.2",
	"/packages/log4j-1.2.16.jar":                                   "pkg:maven/log4j/log4j@1.2.16",
	"/packages/log4j-core-2.17.0.jar":                              "pkg:maven/org.apache.logging.log4j/log4j-core@2.17.0",
	"/packages/log4j-over-slf4j-1.7.36.jar":                        "pkg:maven/org.slf4j/log4j-over-slf4j@1.7.36",
	"/packages/manager-pojo-1.8.0.jar":                             "pkg:maven/org.apache.inlong/manager-pojo@1.8.0",
	"/packages/maven-plugin.hpi":                                   "pkg:maven/org.jenkins-ci.main/maven-plugin@3.10.1",
	"/packages/maven-plugin.hpi:WEB-INF/lib/annotations-3.0.1.jar": "pkg:maven/com.google.code.findbugs/annotations@3.0.1",
	"/packages/maven-plugin.hpi:WEB-INF/lib/annotations-3.0.1.jar:com.google.code.findbugs:jsr305": "pkg:maven/com.google.code.findbugs/jsr305@3.0.1",
	"/packages/maven-plugin.hpi:WEB-INF/lib/cdi-api-1.0.jar":                                       "pkg:maven/javax.enterprise/cdi-api@1.0",
	"/packages/maven-plugin.hpi:WEB-INF/lib/commons-cli-1.4.jar":                                   "pkg:maven/commons-cli/commons-cli@1.4",
	"/packages/maven-plugin.hpi:WEB-INF/lib/commons-lang3-3.11.jar":                                "pkg:maven/org.apache.commons/commons-lang3@3.11",
	"/packages/maven-plugin.hpi:WEB-INF/lib/commons-net-3.6.jar":                                   "pkg:maven/commons-net/commons-net@3.6",
	"/packages/maven-plugin.hpi:WEB-INF/lib/doxia-sink-api-1.0.jar":                                "pkg:maven/org.apache.maven.doxia/doxia-sink-api@1.0",
	"/packages/maven-plugin.hpi:WEB-INF/lib/jackrabbit-webdav-2.14.4.jar":                          "pkg:maven/org.apache.jackrabbit/jackrabbit-webdav@2.14.4",
	"/packages/maven-plugin.hpi:WEB-INF/lib/jcl-over-slf4j-1.7.30.jar":                             "pkg:maven/org.slf4j/jcl-over-slf4j@1.7.30",
	"/packages/maven-plugin.hpi:WEB-INF/lib/jna-platform-4.1.0.jar":                                "pkg:maven/com.sun.jna.platform/jna-platform@4.1.0",
	"/packages/maven-plugin.hpi:WEB-INF/lib/jsch.agentproxy.connector-factory-0.0.9.jar":           "pkg:maven/com.jcraft/jsch.agentproxy.connector-factory@0.0.9",
	"/packages/maven-plugin.hpi:WEB-INF/lib/jsch.agentproxy.core-0.0.9.jar":                        "pkg:maven/com.jcraft/jsch.agentproxy.core@0.0.9",
	"/packages/maven-plugin.hpi:WEB-INF/lib/jsch.agentproxy.jsch-0.0.9.jar":                        "pkg:maven/com.jcraft/jsch.agentproxy.jsch@0.0.9",
	"/packages/maven-plugin.hpi:WEB-INF/lib/jsch.agentproxy.pageant-0.0.9.jar":                     "pkg:maven/com.jcraft/jsch.agentproxy.pageant@0.0.9",
	"/packages/maven-plugin.hpi:WEB-INF/lib/jsch.agentproxy.sshagent-0.0.9.jar":                    "pkg:maven/com.jcraft/jsch.agentproxy.sshagent@0.0.9",
	"/packages/maven-plugin.hpi:WEB-INF/lib/jsch.agentproxy.usocket-jna-0.0.9.jar":                 "pkg:maven/com.jcraft/jsch.agentproxy.usocket-jna@0.0.9",
	"/packages/maven-plugin.hpi:WEB-INF/lib/jsch.agentproxy.usocket-nc-0.0.9.jar":                  "pkg:maven/com.jcraft/jsch.agentproxy.usocket-nc@0.0.9",
	"/packages/maven-plugin.hpi:WEB-INF/lib/jsoup-1.12.1.jar":                                      "pkg:maven/org.jsoup/jsoup@1.12.1",
	"/packages/maven-plugin.hpi:WEB-INF/lib/lib-jenkins-maven-artifact-manager-1.2.jar":            "pkg:maven/org.jenkins-ci.lib/lib-jenkins-maven-artifact-manager@1.2",
	"/packages/maven-plugin.hpi:WEB-INF/lib/lib-jenkins-maven-embedder-3.15.jar":                   "pkg:maven/org.jenkins-ci.lib/lib-jenkins-maven-embedder@3.15",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven-agent-1.13.jar":                                  "pkg:maven/org.jenkins-ci.main.maven/maven-agent@1.13",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven-artifact-3.5.4.jar":                              "pkg:maven/org.apache.maven/maven-artifact@3.5.4",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven-builder-support-3.5.4.jar":                       "pkg:maven/org.apache.maven/maven-builder-support@3.5.4",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven-compat-3.5.4.jar":                                "pkg:maven/org.apache.maven/maven-compat@3.5.4",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven-core-3.5.4.jar":                                  "pkg:maven/org.apache.maven/maven-core@3.5.4",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven-embedder-3.5.4.jar":                              "pkg:maven/org.apache.maven/maven-embedder@3.5.4",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven-interceptor-1.13.jar":                            "pkg:maven/org.jenkins-ci.main.maven/maven-interceptor@1.13",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven-model-3.5.4.jar":                                 "pkg:maven/org.apache.maven/maven-model@3.5.4",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven-model-builder-3.5.4.jar":                         "pkg:maven/org.apache.maven/maven-model-builder@3.5.4",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven-plugin-api-3.5.4.jar":                            "pkg:maven/org.apache.maven/maven-plugin-api@3.5.4",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven-plugin.jar":                                      "pkg:maven/org.jenkins-ci.main/maven-plugin@3.10.1",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven-plugin.jar:classworlds.jar":                      "pkg:maven/org.codehaus.classworlds/classworlds@1.1",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven-reporting-api-3.0.jar":                           "pkg:maven/org.apache.maven.reporting/maven-reporting-api@3.0",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven-repository-metadata-3.5.4.jar":                   "pkg:maven/org.apache.maven/maven-repository-metadata@3.5.4",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven-resolver-api-1.1.1.jar":                          "pkg:maven/org.apache.maven.resolver/maven-resolver-api@1.1.1",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven-resolver-connector-basic-1.1.1.jar":              "pkg:maven/org.apache.maven.resolver/maven-resolver-connector-basic@1.1.1",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven-resolver-impl-1.1.1.jar":                         "pkg:maven/org.apache.maven.resolver/maven-resolver-impl@1.1.1",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven-resolver-provider-3.5.4.jar":                     "pkg:maven/org.apache.maven/maven-resolver-provider@3.5.4",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven-resolver-spi-1.1.1.jar":                          "pkg:maven/org.apache.maven.resolver/maven-resolver-spi@1.1.1",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven-resolver-transport-wagon-1.1.1.jar":              "pkg:maven/org.apache.maven.resolver/maven-resolver-transport-wagon@1.1.1",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven-resolver-util-1.1.1.jar":                         "pkg:maven/org.apache.maven.resolver/maven-resolver-util@1.1.1",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven-settings-3.5.4.jar":                              "pkg:maven/org.apache.maven/maven-settings@3.5.4",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven-settings-builder-3.5.4.jar":                      "pkg:maven/org.apache.maven/maven-settings-builder@3.5.4",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven-shared-utils-3.2.1.jar":                          "pkg:maven/org.apache.maven.shared/maven-shared-utils@3.2.1",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven2.1-interceptor-1.2.jar":                          "pkg:maven/org.jvnet.hudson/maven2.1-interceptor@1.2",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven3-agent-1.13.jar":                                 "pkg:maven/org.jenkins-ci.main.maven/maven3-agent@1.13",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven3-interceptor-1.13.jar":                           "pkg:maven/org.jenkins-ci.main.maven/maven3-interceptor@1.13",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven3-interceptor-commons-1.13.jar":                   "pkg:maven/org.jenkins-ci.main.maven/maven3-interceptor-commons@1.13",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven31-agent-1.13.jar":                                "pkg:maven/org.jenkins-ci.main.maven/maven31-agent@1.13",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven31-interceptor-1.13.jar":                          "pkg:maven/org.jenkins-ci.main.maven/maven31-interceptor@1.13",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven32-agent-1.13.jar":                                "pkg:maven/org.jenkins-ci.main.maven/maven32-agent@1.13",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven32-interceptor-1.13.jar":                          "pkg:maven/org.jenkins-ci.main.maven/maven32-interceptor@1.13",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven33-agent-1.13.jar":                                "pkg:maven/org.jenkins-ci.main.maven/maven33-agent@1.13",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven33-interceptor-1.13.jar":                          "pkg:maven/org.jenkins-ci.main.maven/maven33-interceptor@1.13",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven35-agent-1.13.jar":                                "pkg:maven/org.jenkins-ci.main.maven/maven35-agent@1.13",
	"/packages/maven-plugin.hpi:WEB-INF/lib/maven35-interceptor-1.13.jar":                          "pkg:maven/org.jenkins-ci.main.maven/maven35-interceptor@1.13",
	"/packages/maven-plugin.hpi:WEB-INF/lib/org.eclipse.sisu.inject-0.3.3.jar":                     "pkg:maven/org.eclipse.sisu.inject/org.eclipse.sisu.inject@0.3.3",
	"/packages/maven-plugin.hpi:WEB-INF/lib/org.eclipse.sisu.plexus-0.3.3.jar":                     "pkg:maven/org.eclipse.sisu.plexus/org.eclipse.sisu.plexus@0.3.3",
	"/packages/maven-plugin.hpi:WEB-INF/lib/plexus-cipher-1.8.jar":                                 "pkg:maven/org.codehaus.plexus/plexus-cipher@1.8",
	"/packages/maven-plugin.hpi:WEB-INF/lib/plexus-classworlds-2.6.0.jar":                          "pkg:maven/org.codehaus.plexus/plexus-classworlds@2.6.0",
	"/packages/maven-plugin.hpi:WEB-INF/lib/plexus-component-annotations-2.1.0.jar":                "pkg:maven/org.codehaus.plexus/plexus-component-annotations@2.1.0",
	"/packages/maven-plugin.hpi:WEB-INF/lib/plexus-interactivity-api-1.0.jar":                      "pkg:maven/org.codehaus.plexus/plexus-interactivity-api@1.0",
	"/packages/maven-plugin.hpi:WEB-INF/lib/plexus-interpolation-1.24.jar":                         "pkg:maven/org.codehaus.plexus/plexus-interpolation@1.24",
	"/packages/maven-plugin.hpi:WEB-INF/lib/plexus-sec-dispatcher-1.4.jar":                         "pkg:maven/org.sonatype.plexus/plexus-sec-dispatcher@1.4",
	"/packages/maven-plugin.hpi:WEB-INF/lib/plexus-utils-3.3.0.jar":                                "pkg:maven/org.codehaus.plexus/plexus-utils@3.3.0",
	"/packages/maven-plugin.hpi:WEB-INF/lib/slf4j-api-1.7.30.jar":                                  "pkg:maven/org.slf4j/slf4j-api@1.7.30",
	"/packages/maven-plugin.hpi:WEB-INF/lib/wagon-file-3.4.3.jar":                                  "pkg:maven/org.apache.maven.wagon/wagon-file@3.4.3",
	"/packages/maven-plugin.hpi:WEB-INF/lib/wagon-ftp-3.4.3.jar":                                   "pkg:maven/org.apache.maven.wagon/wagon-ftp@3.4.3",
	"/packages/maven-plugin.hpi:WEB-INF/lib/wagon-http-3.4.3.jar":                                  "pkg:maven/org.apache.maven.wagon/wagon-http@3.4.3",
	"/packages/maven-plugin.hpi:WEB-INF/lib/wagon-http-shared-3.4.3.jar":                           "pkg:maven/org.apache.maven.wagon/wagon-http-shared@3.4.3",
	"/packages/maven-plugin.hpi:WEB-INF/lib/wagon-provider-api-3.4.3.jar":                          "pkg:maven/org.apache.maven.wagon/wagon-provider-api@3.4.3",
	"/packages/maven-plugin.hpi:WEB-INF/lib/wagon-ssh-3.4.3.jar":                                   "pkg:maven/org.apache.maven.wagon/wagon-ssh@3.4.3",
	"/packages/maven-plugin.hpi:WEB-INF/lib/wagon-ssh-common-3.4.3.jar":                            "pkg:maven/org.apache.maven.wagon/wagon-ssh-common@3.4.3",
	"/packages/maven-plugin.hpi:WEB-INF/lib/wagon-ssh-external-3.4.3.jar":                          "pkg:maven/org.apache.maven.wagon/wagon-ssh-external@3.4.3",
	"/packages/maven-plugin.hpi:WEB-INF/lib/wagon-webdav-jackrabbit-3.4.3.jar":                     "pkg:maven/org.apache.maven.wagon/wagon-webdav-jackrabbit@3.4.3",
	"/packages/maven-shared-utils-3.2.1.jar":                                                       "pkg:maven/org.apache.maven.shared/maven-shared-utils@3.2.1",
	"/packages/mesos-1.7.1.jar":                                                                    "pkg:maven/org.apache.mesos/mesos@1.7.1",
	"/packages/minio-8.3.8.jar":                                                                    "pkg:maven/io.minio/minio@8.3.8",
	"/packages/ms-mcms-5.3.1.jar":                                                                  "pkg:maven/net.mingsoft/ms-mcms@5.3.1",
	"/packages/my-app-1.jar":                                                                       "pkg:maven/com.mycompany.app/my-app@1",
	"/packages/my-app-1.jar:com.fasterxml.jackson.core:jackson-annotations":                        "pkg:maven/com.fasterxml.jackson.core/jackson-annotations@2.13.2",
	"/packages/my-app-1.jar:com.fasterxml.jackson.core:jackson-core":                               "pkg:maven/com.fasterxml.jackson.core/jackson-core@2.13.2",
	"/packages/my-app-1.jar:com.fasterxml.jackson.core:jackson-databind":                           "pkg:maven/com.fasterxml.jackson.core/jackson-databind@2.13.0",
	"/packages/my-app-1.jar:com.fasterxml:classmate":                                               "pkg:maven/com.fasterxml/classmate@1.5.1",
	"/packages/my-app-1.jar:com.google.errorprone:error_prone_annotations":                         "pkg:maven/com.google.errorprone/error_prone_annotations@2.0.18",
	"/packages/my-app-1.jar:com.google.guava:guava":                                                "pkg:maven/com.google.guava/guava@23.0",
	"/packages/my-app-1.jar:com.google.j2objc:j2objc-annotations":                                  "pkg:maven/com.google.j2objc/j2objc-annotations@1.1",
	"/packages/my-app-1.jar:com.hazelcast:hazelcast-kubernetes":                                    "pkg:maven/com.hazelcast/hazelcast-kubernetes@1.5.4",
	"/packages/my-app-1.jar:com.hierynomus:asn-one":                                                "pkg:maven/com.hierynomus/asn-one@0.5.0",
	"/packages/my-app-1.jar:com.typesafe.netty:netty-reactive-streams":                             "pkg:maven/com.typesafe.netty/netty-reactive-streams@2.0.5",
	"/packages/my-app-1.jar:commons-beanutils:commons-beanutils":                                   "pkg:maven/commons-beanutils/commons-beanutils@1.9.2",
	"/packages/my-app-1.jar:commons-collections:commons-collections":                               "pkg:maven/commons-collections/commons-collections@3.2.2",
	"/packages/my-app-1.jar:commons-digester:commons-digester":                                     "pkg:maven/commons-digester/commons-digester@1.8.1",
	"/packages/my-app-1.jar:commons-io:commons-io":                                                 "pkg:maven/commons-io/commons-io@2.8.0",
	"/packages/my-app-1.jar:commons-logging:commons-logging":                                       "pkg:maven/commons-logging/commons-logging@1.2",
	"/packages/my-app-1.jar:commons-validator:commons-validator":                                   "pkg:maven/commons-validator/commons-validator@1.5.1",
	"/packages/my-app-1.jar:io.netty:netty-buffer":                                                 "pkg:maven/io.netty/netty-buffer@4.1.52.Final",
	"/packages/my-app-1.jar:io.netty:netty-codec":                                                  "pkg:maven/io.netty/netty-codec@4.1.52.Final",
	"/packages/my-app-1.jar:io.netty:netty-codec-dns":                                              "pkg:maven/io.netty/netty-codec-dns@4.1.63.Final",
	"/packages/my-app-1.jar:io.netty:netty-codec-http":                                             "pkg:maven/io.netty/netty-codec-http@4.1.63.Final",
	"/packages/my-app-1.jar:io.netty:netty-codec-socks":                                            "pkg:maven/io.netty/netty-codec-socks@4.1.63.Final",
	"/packages/my-app-1.jar:io.netty:netty-common":                                                 "pkg:maven/io.netty/netty-common@4.1.52.Final",
	"/packages/my-app-1.jar:io.netty:netty-handler":                                                "pkg:maven/io.netty/netty-handler@4.1.52.Final",
	"/packages/my-app-1.jar:io.netty:netty-handler-proxy":                                          "pkg:maven/io.netty/netty-handler-proxy@4.1.63.Final",
	"/packages/my-app-1.jar:io.netty:netty-resolver":                                               "pkg:maven/io.netty/netty-resolver@4.1.52.Final",
	"/packages/my-app-1.jar:io.netty:netty-resolver-dns":                                           "pkg:maven/io.netty/netty-resolver-dns@4.1.63.Final",
	"/packages/my-app-1.jar:io.netty:netty-resolver-dns-native-macos":                              "pkg:maven/io.netty/netty-resolver-dns-native-macos@4.1.63.Final",
	"/packages/my-app-1.jar:io.netty:netty-transport":                                              "pkg:maven/io.netty/netty-transport@4.1.52.Final",
	"/packages/my-app-1.jar:io.netty:netty-transport-native-epoll":                                 "pkg:maven/io.netty/netty-transport-native-epoll@4.1.63.Final",
	"/packages/my-app-1.jar:io.netty:netty-transport-native-unix-common":                           "pkg:maven/io.netty/netty-transport-native-unix-common@4.1.63.Final",
	"/packages/my-app-1.jar:io.prometheus:simpleclient":                                            "pkg:maven/io.prometheus/simpleclient@0.9.0",
	"/packages/my-app-1.jar:io.prometheus:simpleclient_common":                                     "pkg:maven/io.prometheus/simpleclient_common@0.9.0",
	"/packages/my-app-1.jar:io.vavr:vavr":                                                          "pkg:maven/io.vavr/vavr@0.10.2",
	"/packages/my-app-1.jar:io.vavr:vavr-match":                                                    "pkg:maven/io.vavr/vavr-match@0.10.2",
	"/packages/my-app-1.jar:jakarta.validation:jakarta.validation-api":                             "pkg:maven/jakarta.validation/jakarta.validation-api@2.0.2",
	"/packages/my-app-1.jar:javax.servlet:javax.servlet-api":                                       "pkg:maven/javax.servlet/javax.servlet-api@3.1.0",
	"/packages/my-app-1.jar:org.apache.activemq.protobuf:activemq-protobuf":                        "pkg:maven/org.apache.activemq.protobuf/activemq-protobuf@1.1",
	"/packages/my-app-1.jar:org.apache.commons:commons-compress":                                   "pkg:maven/org.apache.commons/commons-compress@1.21",
	"/packages/my-app-1.jar:org.apache.commons:commons-lang3":                                      "pkg:maven/org.apache.commons/commons-lang3@3.10",
	"/packages/my-app-1.jar:org.apache.maven.shared:maven-shared-utils":                            "pkg:maven/org.apache.maven.shared/maven-shared-utils@3.3.4",
	"/packages/my-app-1.jar:org.codehaus.mojo:animal-sniffer-annotations":                          "pkg:maven/org.codehaus.mojo/animal-sniffer-annotations@1.14",
	"/packages/my-app-1.jar:org.eclipse.jetty.http2:http2-common":                                  "pkg:maven/org.eclipse.jetty.http2/http2-common@9.4.43.v20210629",
	"/packages/my-app-1.jar:org.eclipse.jetty.http2:http2-hpack":                                   "pkg:maven/org.eclipse.jetty.http2/http2-hpack@9.4.43.v20210629",
	"/packages/my-app-1.jar:org.eclipse.jetty.toolchain.setuid:jetty-setuid-java":                  "pkg:maven/org.eclipse.jetty.toolchain.setuid/jetty-setuid-java@1.0.4",
	"/packages/my-app-1.jar:org.eclipse.jetty:jetty-client":                                        "pkg:maven/org.eclipse.jetty/jetty-client@9.4.43.v20210629",
	"/packages/my-app-1.jar:org.eclipse.jetty:jetty-http":                                          "pkg:maven/org.eclipse.jetty/jetty-http@9.4.43.v20210629",
	"/packages/my-app-1.jar:org.eclipse.jetty:jetty-io":                                            "pkg:maven/org.eclipse.jetty/jetty-io@9.4.43.v20210629",
	"/packages/my-app-1.jar:org.eclipse.jetty:jetty-server":                                        "pkg:maven/org.eclipse.jetty/jetty-server@9.1.0.M0",
	"/packages/my-app-1.jar:org.eclipse.jetty:jetty-util":                                          "pkg:maven/org.eclipse.jetty/jetty-util@9.4.43.v20210629",
	"/packages/my-app-1.jar:org.everit.json:org.everit.json.schema":                                "pkg:maven/org.everit.json/org.everit.json.schema@1.5.1",
	"/packages/my-app-1.jar:org.hibernate.validator:hibernate-validator":                           "pkg:maven/org.hibernate.validator/hibernate-validator@6.2.4.Final",
	"/packages/my-app-1.jar:org.jboss.logging:jboss-logging":                                       "pkg:maven/org.jboss.logging/jboss-logging@3.4.1.Final",
	"/packages/my-app-1.jar:org.jctools:jctools-core":                                              "pkg:maven/org.jctools/jctools-core@3.1.0",
	"/packages/my-app-1.jar:org.jetbrains:annotations":                                             "pkg:maven/org.jetbrains/annotations@13.0",
	"/packages/my-app-1.jar:org.json:json":                                                         "pkg:maven/org.json/json@20160810",
	"/packages/my-app-1.jar:org.kohsuke:github-api":                                                "pkg:maven/org.kohsuke/github-api@1.301",
	"/packages/my-app-1.jar:org.slf4j:log4j-over-slf4j":                                            "pkg:maven/org.slf4j/log4j-over-slf4j@1.7.33",
	"/packages/my-app-1.jar:org.slf4j:slf4j-api":                                                   "pkg:maven/org.slf4j/slf4j-api@1.7.33",
	"/packages/my-app-1.jar:org.sonarsource.analyzer-commons:sonar-analyzer-test-commons":          "pkg:maven/org.sonarsource.analyzer-commons/sonar-analyzer-test-commons@1.14.1.690",
	"/packages/my-app-1.jar:org.sonarsource.php:php-checks":                                        "pkg:maven/org.sonarsource.php/php-checks@3.17.0.7439",
	"/packages/my-app-1.jar:org.sonarsource.php:php-frontend":                                      "pkg:maven/org.sonarsource.php/php-frontend@3.17.0.7439",
	"/packages/my-app-1.jar:org.sonarsource.python:python-checks":                                  "pkg:maven/org.sonarsource.python/python-checks@3.4.1.8066",
	"/packages/my-app-1.jar:org.sonarsource.python:python-frontend":                                "pkg:maven/org.sonarsource.python/python-frontend@3.4.1.8066",
	"/packages/my-app-1.jar:org.sonarsource.sslr:sslr-core":                                        "pkg:maven/org.sonarsource.sslr/sslr-core@1.24.0.633",
	"/packages/netty-reactive-streams-2.0.6.jar":                                                   "pkg:maven/com.typesafe.netty/netty-reactive-streams@2.0.6",
	"/packages/nifi-utils-1.12.0.jar":                                                              "pkg:maven/org.apache.nifi/nifi-utils@1.12.0",
	"/packages/nomad.hpi":                                                                          "pkg:maven/org.jenkins-ci.plugins/nomad@0.7.4",
	"/packages/nomad.hpi:WEB-INF/lib/annotations-13.0.jar":                                         "pkg:maven/org.jetbrains/annotations@13.0",
	"/packages/nomad.hpi:WEB-INF/lib/gson-2.8.6.jar":                                               "pkg:maven/com.google.code.gson/gson@2.8.6",
	"/packages/nomad.hpi:WEB-INF/lib/json-20200518.jar":                                            "pkg:maven/org.json/json@20200518",
	"/packages/nomad.hpi:WEB-INF/lib/kotlin-stdlib-1.3.70.jar":                                     "pkg:maven/kotlin-stdlib/kotlin-stdlib@1.3.70",
	"/packages/nomad.hpi:WEB-INF/lib/kotlin-stdlib-common-1.3.70.jar":                              "pkg:maven/kotlin-stdlib-common/kotlin-stdlib-common@1.3.70",
	"/packages/nomad.hpi:WEB-INF/lib/nomad.jar":                                                    "pkg:maven/org.jenkins-ci.plugins/nomad@0.7.4",
	"/packages/nomad.hpi:WEB-INF/lib/okhttp-4.5.0.jar":                                             "pkg:maven/com.squareup.okhttp3/okhttp@4.5.0",
	"/packages/nomad.hpi:WEB-INF/lib/okio-2.5.0.jar":                                               "pkg:maven/com.squareup.okio/okio@2.5.0",
	"/packages/nuget.hpi": "pkg:maven/org.jenkins-ci.plugins/nuget@1.0",
	"/packages/nuget.hpi:WEB-INF/lib/envinject-lib-1.19.jar":                                                       "pkg:maven/org.jenkins-ci.lib/envinject-lib@1.19",
	"/packages/nuget.hpi:WEB-INF/lib/nuget.jar":                                                                    "pkg:maven/org.jenkins-ci.plugins/nuget@1.0",
	"/packages/nuget.hpi:WEB-INF/lib/xtrigger-lib-0.33.jar":                                                        "pkg:maven/org.jenkins-ci.lib/xtrigger-lib@0.33",
	"/packages/openmeetings-util-4.0.9.jar":                                                                        "pkg:maven/org.apache.openmeetings/openmeetings-util@4.0.9",
	"/packages/org.eclipse.ant.core-3.7.0.jar":                                                                     "pkg:maven/org.eclipse.platform/org.eclipse.ant.core@3.7.0",
	"/packages/org.eclipse.osgi-3.18.0.jar":                                                                        "pkg:maven/org.eclipse.platform/org.eclipse.osgi@3.18.0",
	"/packages/org.everit.json.schema-1.5.1.jar":                                                                   "pkg:maven/org.everit.json/org.everit.json.schema@1.5.1",
	"/packages/original-my-app-1.jar":                                                                              "pkg:maven/com.mycompany.app/my-app@1",
	"/packages/original-my-app-1.jar:com.fasterxml.jackson.core:jackson-annotations":                               "pkg:maven/com.fasterxml.jackson.core/jackson-annotations@2.13.2",
	"/packages/original-my-app-1.jar:com.fasterxml.jackson.core:jackson-core":                                      "pkg:maven/com.fasterxml.jackson.core/jackson-core@2.13.2",
	"/packages/original-my-app-1.jar:com.fasterxml.jackson.core:jackson-databind":                                  "pkg:maven/com.fasterxml.jackson.core/jackson-databind@2.13.0",
	"/packages/original-my-app-1.jar:com.fasterxml:classmate":                                                      "pkg:maven/com.fasterxml/classmate@1.5.1",
	"/packages/original-my-app-1.jar:com.google.errorprone:error_prone_annotations":                                "pkg:maven/com.google.errorprone/error_prone_annotations@2.0.18",
	"/packages/original-my-app-1.jar:com.google.guava:guava":                                                       "pkg:maven/com.google.guava/guava@23.0",
	"/packages/original-my-app-1.jar:com.google.j2objc:j2objc-annotations":                                         "pkg:maven/com.google.j2objc/j2objc-annotations@1.1",
	"/packages/original-my-app-1.jar:com.hazelcast:hazelcast-kubernetes":                                           "pkg:maven/com.hazelcast/hazelcast-kubernetes@1.5.4",
	"/packages/original-my-app-1.jar:com.hierynomus:asn-one":                                                       "pkg:maven/com.hierynomus/asn-one@0.5.0",
	"/packages/original-my-app-1.jar:com.typesafe.netty:netty-reactive-streams":                                    "pkg:maven/com.typesafe.netty/netty-reactive-streams@2.0.5",
	"/packages/original-my-app-1.jar:commons-beanutils:commons-beanutils":                                          "pkg:maven/commons-beanutils/commons-beanutils@1.9.2",
	"/packages/original-my-app-1.jar:commons-collections:commons-collections":                                      "pkg:maven/commons-collections/commons-collections@3.2.2",
	"/packages/original-my-app-1.jar:commons-digester:commons-digester":                                            "pkg:maven/commons-digester/commons-digester@1.8.1",
	"/packages/original-my-app-1.jar:commons-io:commons-io":                                                        "pkg:maven/commons-io/commons-io@2.8.0",
	"/packages/original-my-app-1.jar:commons-logging:commons-logging":                                              "pkg:maven/commons-logging/commons-logging@1.2",
	"/packages/original-my-app-1.jar:commons-validator:commons-validator":                                          "pkg:maven/commons-validator/commons-validator@1.5.1",
	"/packages/original-my-app-1.jar:io.netty:netty-buffer":                                                        "pkg:maven/io.netty/netty-buffer@4.1.52.Final",
	"/packages/original-my-app-1.jar:io.netty:netty-codec":                                                         "pkg:maven/io.netty/netty-codec@4.1.52.Final",
	"/packages/original-my-app-1.jar:io.netty:netty-codec-dns":                                                     "pkg:maven/io.netty/netty-codec-dns@4.1.63.Final",
	"/packages/original-my-app-1.jar:io.netty:netty-codec-http":                                                    "pkg:maven/io.netty/netty-codec-http@4.1.63.Final",
	"/packages/original-my-app-1.jar:io.netty:netty-codec-socks":                                                   "pkg:maven/io.netty/netty-codec-socks@4.1.63.Final",
	"/packages/original-my-app-1.jar:io.netty:netty-common":                                                        "pkg:maven/io.netty/netty-common@4.1.52.Final",
	"/packages/original-my-app-1.jar:io.netty:netty-handler":                                                       "pkg:maven/io.netty/netty-handler@4.1.52.Final",
	"/packages/original-my-app-1.jar:io.netty:netty-handler-proxy":                                                 "pkg:maven/io.netty/netty-handler-proxy@4.1.63.Final",
	"/packages/original-my-app-1.jar:io.netty:netty-resolver":                                                      "pkg:maven/io.netty/netty-resolver@4.1.52.Final",
	"/packages/original-my-app-1.jar:io.netty:netty-resolver-dns":                                                  "pkg:maven/io.netty/netty-resolver-dns@4.1.63.Final",
	"/packages/original-my-app-1.jar:io.netty:netty-resolver-dns-native-macos":                                     "pkg:maven/io.netty/netty-resolver-dns-native-macos@4.1.63.Final",
	"/packages/original-my-app-1.jar:io.netty:netty-transport":                                                     "pkg:maven/io.netty/netty-transport@4.1.52.Final",
	"/packages/original-my-app-1.jar:io.netty:netty-transport-native-epoll":                                        "pkg:maven/io.netty/netty-transport-native-epoll@4.1.63.Final",
	"/packages/original-my-app-1.jar:io.netty:netty-transport-native-unix-common":                                  "pkg:maven/io.netty/netty-transport-native-unix-common@4.1.63.Final",
	"/packages/original-my-app-1.jar:io.prometheus:simpleclient":                                                   "pkg:maven/io.prometheus/simpleclient@0.9.0",
	"/packages/original-my-app-1.jar:io.prometheus:simpleclient_common":                                            "pkg:maven/io.prometheus/simpleclient_common@0.9.0",
	"/packages/original-my-app-1.jar:io.vavr:vavr":                                                                 "pkg:maven/io.vavr/vavr@0.10.2",
	"/packages/original-my-app-1.jar:io.vavr:vavr-match":                                                           "pkg:maven/io.vavr/vavr-match@0.10.2",
	"/packages/original-my-app-1.jar:jakarta.validation:jakarta.validation-api":                                    "pkg:maven/jakarta.validation/jakarta.validation-api@2.0.2",
	"/packages/original-my-app-1.jar:javax.servlet:javax.servlet-api":                                              "pkg:maven/javax.servlet/javax.servlet-api@3.1.0",
	"/packages/original-my-app-1.jar:org.apache.activemq.protobuf:activemq-protobuf":                               "pkg:maven/org.apache.activemq.protobuf/activemq-protobuf@1.1",
	"/packages/original-my-app-1.jar:org.apache.commons:commons-compress":                                          "pkg:maven/org.apache.commons/commons-compress@1.21",
	"/packages/original-my-app-1.jar:org.apache.commons:commons-lang3":                                             "pkg:maven/org.apache.commons/commons-lang3@3.10",
	"/packages/original-my-app-1.jar:org.apache.maven.shared:maven-shared-utils":                                   "pkg:maven/org.apache.maven.shared/maven-shared-utils@3.3.4",
	"/packages/original-my-app-1.jar:org.codehaus.mojo:animal-sniffer-annotations":                                 "pkg:maven/org.codehaus.mojo/animal-sniffer-annotations@1.14",
	"/packages/original-my-app-1.jar:org.eclipse.jetty.http2:http2-common":                                         "pkg:maven/org.eclipse.jetty.http2/http2-common@9.4.43.v20210629",
	"/packages/original-my-app-1.jar:org.eclipse.jetty.http2:http2-hpack":                                          "pkg:maven/org.eclipse.jetty.http2/http2-hpack@9.4.43.v20210629",
	"/packages/original-my-app-1.jar:org.eclipse.jetty.toolchain.setuid:jetty-setuid-java":                         "pkg:maven/org.eclipse.jetty.toolchain.setuid/jetty-setuid-java@1.0.4",
	"/packages/original-my-app-1.jar:org.eclipse.jetty:jetty-client":                                               "pkg:maven/org.eclipse.jetty/jetty-client@9.4.43.v20210629",
	"/packages/original-my-app-1.jar:org.eclipse.jetty:jetty-http":                                                 "pkg:maven/org.eclipse.jetty/jetty-http@9.4.43.v20210629",
	"/packages/original-my-app-1.jar:org.eclipse.jetty:jetty-io":                                                   "pkg:maven/org.eclipse.jetty/jetty-io@9.4.43.v20210629",
	"/packages/original-my-app-1.jar:org.eclipse.jetty:jetty-server":                                               "pkg:maven/org.eclipse.jetty/jetty-server@9.1.0.M0",
	"/packages/original-my-app-1.jar:org.eclipse.jetty:jetty-util":                                                 "pkg:maven/org.eclipse.jetty/jetty-util@9.4.43.v20210629",
	"/packages/original-my-app-1.jar:org.everit.json:org.everit.json.schema":                                       "pkg:maven/org.everit.json/org.everit.json.schema@1.5.1",
	"/packages/original-my-app-1.jar:org.hibernate.validator:hibernate-validator":                                  "pkg:maven/org.hibernate.validator/hibernate-validator@6.2.4.Final",
	"/packages/original-my-app-1.jar:org.jboss.logging:jboss-logging":                                              "pkg:maven/org.jboss.logging/jboss-logging@3.4.1.Final",
	"/packages/original-my-app-1.jar:org.jctools:jctools-core":                                                     "pkg:maven/org.jctools/jctools-core@3.1.0",
	"/packages/original-my-app-1.jar:org.jetbrains:annotations":                                                    "pkg:maven/org.jetbrains/annotations@13.0",
	"/packages/original-my-app-1.jar:org.json:json":                                                                "pkg:maven/org.json/json@20160810",
	"/packages/original-my-app-1.jar:org.kohsuke:github-api":                                                       "pkg:maven/org.kohsuke/github-api@1.301",
	"/packages/original-my-app-1.jar:org.slf4j:log4j-over-slf4j":                                                   "pkg:maven/org.slf4j/log4j-over-slf4j@1.7.33",
	"/packages/original-my-app-1.jar:org.slf4j:slf4j-api":                                                          "pkg:maven/org.slf4j/slf4j-api@1.7.33",
	"/packages/original-my-app-1.jar:org.sonarsource.analyzer-commons:sonar-analyzer-test-commons":                 "pkg:maven/org.sonarsource.analyzer-commons/sonar-analyzer-test-commons@1.14.1.690",
	"/packages/original-my-app-1.jar:org.sonarsource.php:php-checks":                                               "pkg:maven/org.sonarsource.php/php-checks@3.17.0.7439",
	"/packages/original-my-app-1.jar:org.sonarsource.php:php-frontend":                                             "pkg:maven/org.sonarsource.php/php-frontend@3.17.0.7439",
	"/packages/original-my-app-1.jar:org.sonarsource.python:python-checks":                                         "pkg:maven/org.sonarsource.python/python-checks@3.4.1.8066",
	"/packages/original-my-app-1.jar:org.sonarsource.python:python-frontend":                                       "pkg:maven/org.sonarsource.python/python-frontend@3.4.1.8066",
	"/packages/original-my-app-1.jar:org.sonarsource.sslr:sslr-core":                                               "pkg:maven/org.sonarsource.sslr/sslr-core@1.24.0.633",
	"/packages/php-frontend-3.9.0.6331.jar":                                                                        "pkg:maven/org.sonarsource.php/php-frontend@3.9.0.6331",
	"/packages/postgresql-42.5.0.jar":                                                                              "pkg:maven/org.postgresql/postgresql@42.5.0",
	"/packages/protobuf-java-3.21.6.jar":                                                                           "pkg:maven/com.google.protobuf/protobuf-java@3.21.6",
	"/packages/python-frontend-3.24.0.10784.jar":                                                                   "pkg:maven/org.sonarsource.python/python-frontend@3.24.0.10784",
	"/packages/ratpack-core-1.8.2.jar":                                                                             "pkg:maven/io.ratpack/ratpack-core@1.8.2",
	"/packages/reactor-netty-core-1.0.35.jar":                                                                      "pkg:maven/io.projectreactor.netty/reactor-netty-core@1.0.35",
	"/packages/reactor-netty-http-1.1.9.jar":                                                                       "pkg:maven/io.projectreactor.netty/reactor-netty-http@1.1.9",
	"/packages/reactor-netty-incubator-quic-0.1.3.jar":                                                             "pkg:maven/io.projectreactor.netty.incubator/reactor-netty-incubator-quic@0.1.3",
	"/packages/resilience4j-prometheus-0.17.0.jar":                                                                 "pkg:maven/io.github.resilience4j.prometheus/resilience4j-prometheus@0.17.0",
	"/packages/script-security.hpi":                                                                                "pkg:maven/org.jenkins-ci.plugins/script-security@1.74",
	"/packages/script-security.hpi:WEB-INF/lib/caffeine-2.8.2.jar":                                                 "pkg:maven/com.github.ben-manes.caffeine/caffeine@2.8.2",
	"/packages/script-security.hpi:WEB-INF/lib/checker-qual-3.3.0.jar":                                             "pkg:maven/org.checkerframework.checker.qual/checker-qual@3.3.0",
	"/packages/script-security.hpi:WEB-INF/lib/error_prone_annotations-2.3.4.jar":                                  "pkg:maven/com.google.errorprone/error_prone_annotations@2.3.4",
	"/packages/script-security.hpi:WEB-INF/lib/groovy-sandbox-1.26.jar":                                            "pkg:maven/org.kohsuke/groovy-sandbox@1.26",
	"/packages/script-security.hpi:WEB-INF/lib/script-security.jar":                                                "pkg:maven/org.jenkins-ci.plugins/script-security@1.74",
	"/packages/shiro-core-1.9.1.jar":                                                                               "pkg:maven/org.apache.shiro/shiro-core@1.9.1",
	"/packages/solr-core-8.8.1.jar":                                                                                "pkg:maven/org.apache.solr/solr-core@8.8.1",
	"/packages/spring-amqp-2.4.17.jar":                                                                             "pkg:maven/org.springframework.amqp/spring-amqp@2.4.17",
	"/packages/spring-analytics-2.0.0.RELEASE.jar":                                                                 "pkg:maven/org.springframework.analytics/spring-analytics@2.0.0.RELEASE",
	"/packages/spring-asm-3.0.7.RELEASE.jar":                                                                       "pkg:maven/org.springframework/spring-asm@3.0.7.RELEASE",
	"/packages/spring-batch-core-4.2.1.RELEASE.jar":                                                                "pkg:maven/org.springframework.batch/spring-batch-core@4.2.1.RELEASE",
	"/packages/spring-boot-3.1.2.jar":                                                                              "pkg:maven/org.springframework.boot/spring-boot@3.1.2",
	"/packages/spring-cloud-app-broker-core-1.6.1.jar":                                                             "pkg:maven/org.springframework.cloud/spring-cloud-app-broker-core@1.6.1",
	"/packages/spring-cql-1.5.11.RELEASE.jar":                                                                      "pkg:maven/org.springframework.data/spring-cql@1.5.11.RELEASE",
	"/packages/spring-credhub-core-1.0.1.RELEASE.jar":                                                              "pkg:maven/org.springframework.credhub/spring-credhub-core@1.0.1.RELEASE",
	"/packages/spring-flex-1.0.3.RELEASE.jar":                                                                      "pkg:maven/org.springframework.flex/spring-flex@1.0.3.RELEASE",
	"/packages/spring-graphql-1.2.2.jar":                                                                           "pkg:maven/org.springframework.graphql/spring-graphql@1.2.2",
	"/packages/spring-hateoas-2.1.0.jar":                                                                           "pkg:maven/org.springframework.hateoas/spring-hateoas@2.1.0",
	"/packages/spring-integration-amqp-6.0.5.jar":                                                                  "pkg:maven/org.springframework.integration/spring-integration-amqp@6.0.5",
	"/packages/spring-kafka-3.0.9.jar":                                                                             "pkg:maven/org.springframework.kafka/spring-kafka@3.0.9",
	"/packages/spring-security-core-6.0.2.jar":                                                                     "pkg:maven/org.springframework.security/spring-security-core@6.0.2",
	"/packages/spring-security-kerberos-core-1.0.1.RELEASE.jar":                                                    "pkg:maven/org.springframework.security.kerberos/spring-security-kerberos-core@1.0.1.RELEASE",
	"/packages/spring-security-oauth-2.5.1.RELEASE.jar":                                                            "pkg:maven/org.springframework.security.oauth/spring-security-oauth@2.5.1.RELEASE",
	"/packages/spring-security-oauth2-autoconfigure-2.5.9.jar":                                                     "pkg:maven/org.springframework.security.oauth.boot/spring-security-oauth2-autoconfigure@2.5.9",
	"/packages/spring-security-saml2-core-1.0.10.RELEASE.jar":                                                      "pkg:maven/org.springframework.security.extensions/spring-security-saml2-core@1.0.10.RELEASE",
	"/packages/spring-session-core-3.0.0.jar":                                                                      "pkg:maven/org.springframework.session/spring-session-core@3.0.0",
	"/packages/spring-social-core-1.1.2.RELEASE.jar":                                                               "pkg:maven/org.springframework.social/spring-social-core@1.1.2.RELEASE",
	"/packages/spring-vault-core-3.0.1.jar":                                                                        "pkg:maven/org.springframework.vault/spring-vault-core@3.0.1",
	"/packages/spring-webmvc-3.1.4.RELEASE.jar":                                                                    "pkg:maven/org.springframework/spring-webmvc@3.1.4.RELEASE",
	"/packages/spring-xml-3.0.3.RELEASE.jar":                                                                       "pkg:maven/org.springframework.ws/spring-xml@3.0.3.RELEASE",
	"/packages/ssh-credentials.hpi":                                                                                "pkg:maven/org.jenkins-ci.plugins/ssh-credentials@1.18.1",
	"/packages/ssh-credentials.hpi:WEB-INF/lib/ssh-credentials.jar":                                                "pkg:maven/org.jenkins-ci.plugins/ssh-credentials@1.18.1",
	"/packages/storm-core-1.2.2.jar":                                                                               "pkg:maven/org.apache.storm/storm-core@1.2.2",
	"/packages/storm-core-1.2.2.jar:cheshire:cheshire":                                                             "pkg:maven/cheshire/cheshire@5.3.1",
	"/packages/storm-core-1.2.2.jar:clj-time:clj-time":                                                             "pkg:maven/clj-time/clj-time@0.8.0",
	"/packages/storm-core-1.2.2.jar:clout:clout":                                                                   "pkg:maven/clout/clout@1.2.0",
	"/packages/storm-core-1.2.2.jar:com.fasterxml.jackson.core:jackson-core":                                       "pkg:maven/com.fasterxml.jackson.core/jackson-core@2.9.4",
	"/packages/storm-core-1.2.2.jar:com.fasterxml.jackson.dataformat:jackson-dataformat-smile":                     "pkg:maven/com.fasterxml.jackson.dataformat/jackson-dataformat-smile@2.9.4",
	"/packages/storm-core-1.2.2.jar:com.google.guava:guava":                                                        "pkg:maven/com.google.guava/guava@16.0.1",
	"/packages/storm-core-1.2.2.jar:com.twitter:carbonite":                                                         "pkg:maven/com.twitter/carbonite@1.5.0",
	"/packages/storm-core-1.2.2.jar:commons-codec:commons-codec":                                                   "pkg:maven/commons-codec/commons-codec@1.6",
	"/packages/storm-core-1.2.2.jar:commons-collections:commons-collections":                                       "pkg:maven/commons-collections/commons-collections@3.2.2",
	"/packages/storm-core-1.2.2.jar:commons-fileupload:commons-fileupload":                                         "pkg:maven/commons-fileupload/commons-fileupload@1.3.2",
	"/packages/storm-core-1.2.2.jar:commons-io:commons-io":                                                         "pkg:maven/commons-io/commons-io@2.5",
	"/packages/storm-core-1.2.2.jar:commons-lang:commons-lang":                                                     "pkg:maven/commons-lang/commons-lang@2.5",
	"/packages/storm-core-1.2.2.jar:compojure:compojure":                                                           "pkg:maven/compojure/compojure@1.1.9",
	"/packages/storm-core-1.2.2.jar:hiccup:hiccup":                                                                 "pkg:maven/hiccup/hiccup@0.3.6",
	"/packages/storm-core-1.2.2.jar:io.netty:netty":                                                                "pkg:maven/io.netty/netty@3.9.9.Final",
	"/packages/storm-core-1.2.2.jar:joda-time:joda-time":                                                           "pkg:maven/joda-time/joda-time@2.3",
	"/packages/storm-core-1.2.2.jar:metrics-clojure:metrics-clojure":                                               "pkg:maven/metrics-clojure/metrics-clojure@2.5.1",
	"/packages/storm-core-1.2.2.jar:ns-tracker:ns-tracker":                                                         "pkg:maven/ns-tracker/ns-tracker@0.2.2",
	"/packages/storm-core-1.2.2.jar:org.apache.commons:commons-compress":                                           "pkg:maven/org.apache.commons/commons-compress@1.4.1",
	"/packages/storm-core-1.2.2.jar:org.apache.commons:commons-exec":                                               "pkg:maven/org.apache.commons/commons-exec@1.1",
	"/packages/storm-core-1.2.2.jar:org.apache.curator:curator-client":                                             "pkg:maven/org.apache.curator/curator-client@4.0.1",
	"/packages/storm-core-1.2.2.jar:org.apache.curator:curator-framework":                                          "pkg:maven/org.apache.curator/curator-framework@4.0.1",
	"/packages/storm-core-1.2.2.jar:org.apache.curator:curator-recipes":                                            "pkg:maven/org.apache.curator/curator-recipes@4.0.1",
	"/packages/storm-core-1.2.2.jar:org.apache.hadoop:hadoop-auth":                                                 "pkg:maven/org.apache.hadoop/hadoop-auth@2.6.1",
	"/packages/storm-core-1.2.2.jar:org.apache.httpcomponents:httpclient":                                          "pkg:maven/org.apache.httpcomponents/httpclient@4.3.3",
	"/packages/storm-core-1.2.2.jar:org.apache.httpcomponents:httpcore":                                            "pkg:maven/org.apache.httpcomponents/httpcore@4.4.1",
	"/packages/storm-core-1.2.2.jar:org.clojure:java.jmx":                                                          "pkg:maven/org.clojure/java.jmx@0.3.1",
	"/packages/storm-core-1.2.2.jar:org.clojure:math.numeric-tower":                                                "pkg:maven/org.clojure/math.numeric-tower@0.0.1",
	"/packages/storm-core-1.2.2.jar:org.clojure:tools.cli":                                                         "pkg:maven/org.clojure/tools.cli@0.2.4",
	"/packages/storm-core-1.2.2.jar:org.clojure:tools.logging":                                                     "pkg:maven/org.clojure/tools.logging@0.2.3",
	"/packages/storm-core-1.2.2.jar:org.clojure:tools.macro":                                                       "pkg:maven/org.clojure/tools.macro@0.1.0",
	"/packages/storm-core-1.2.2.jar:org.clojure:tools.namespace":                                                   "pkg:maven/org.clojure/tools.namespace@0.2.4",
	"/packages/storm-core-1.2.2.jar:org.eclipse.jetty:jetty-client":                                                "pkg:maven/org.eclipse.jetty/jetty-client@7.6.13.v20130916",
	"/packages/storm-core-1.2.2.jar:org.eclipse.jetty:jetty-continuation":                                          "pkg:maven/org.eclipse.jetty/jetty-continuation@7.6.13.v20130916",
	"/packages/storm-core-1.2.2.jar:org.eclipse.jetty:jetty-http":                                                  "pkg:maven/org.eclipse.jetty/jetty-http@7.6.13.v20130916",
	"/packages/storm-core-1.2.2.jar:org.eclipse.jetty:jetty-io":                                                    "pkg:maven/org.eclipse.jetty/jetty-io@7.6.13.v20130916",
	"/packages/storm-core-1.2.2.jar:org.eclipse.jetty:jetty-security":                                              "pkg:maven/org.eclipse.jetty/jetty-security@7.6.13.v20130916",
	"/packages/storm-core-1.2.2.jar:org.eclipse.jetty:jetty-server":                                                "pkg:maven/org.eclipse.jetty/jetty-server@7.6.13.v20130916",
	"/packages/storm-core-1.2.2.jar:org.eclipse.jetty:jetty-servlet":                                               "pkg:maven/org.eclipse.jetty/jetty-servlet@7.6.13.v20130916",
	"/packages/storm-core-1.2.2.jar:org.eclipse.jetty:jetty-servlets":                                              "pkg:maven/org.eclipse.jetty/jetty-servlets@7.6.13.v20130916",
	"/packages/storm-core-1.2.2.jar:org.eclipse.jetty:jetty-util":                                                  "pkg:maven/org.eclipse.jetty/jetty-util@7.6.13.v20130916",
	"/packages/storm-core-1.2.2.jar:org.jgrapht:jgrapht-core":                                                      "pkg:maven/org.jgrapht/jgrapht-core@0.9.0",
	"/packages/storm-core-1.2.2.jar:org.yaml:snakeyaml":                                                            "pkg:maven/org.yaml/snakeyaml@1.11",
	"/packages/storm-core-1.2.2.jar:ring:ring-codec":                                                               "pkg:maven/ring/ring-codec@1.0.0",
	"/packages/storm-core-1.2.2.jar:ring:ring-core":                                                                "pkg:maven/ring/ring-core@1.3.1",
	"/packages/storm-core-1.2.2.jar:ring:ring-devel":                                                               "pkg:maven/ring/ring-devel@1.3.1",
	"/packages/storm-core-1.2.2.jar:ring:ring-jetty-adapter":                                                       "pkg:maven/ring/ring-jetty-adapter@1.3.1",
	"/packages/storm-core-1.2.2.jar:ring:ring-json":                                                                "pkg:maven/ring/ring-json@0.3.1",
	"/packages/storm-core-1.2.2.jar:ring:ring-servlet":                                                             "pkg:maven/ring/ring-servlet@1.3.1",
	"/packages/tapestry-core-5.8.1.jar":                                                                            "pkg:maven/org.apache.tapestry/tapestry-core@5.8.1",
	"/packages/tika-core-1.26.jar":                                                                                 "pkg:maven/org.apache.tika/tika-core@1.26",
	"/packages/tomcat-catalina-11.0.0-M1.jar":                                                                      "pkg:maven/org.apache.tomcat/tomcat-catalina@11.0.0-M1",
	"/packages/tomcat-embed-core-11.0.0-M7.jar":                                                                    "pkg:maven/org.apache.tomcat.embed/tomcat-embed-core@11.0.0-M7",
	"/packages/TwilioNotifier.hpi":                                                                                 "pkg:maven/com.twilio.jenkins/TwilioNotifier@0.2.1",
	"/packages/TwilioNotifier.hpi:WEB-INF/lib/commons-codec-1.2.jar":                                               "pkg:maven/commons-codec/commons-codec@1.2",
	"/packages/TwilioNotifier.hpi:WEB-INF/lib/commons-logging-1.0.4.jar":                                           "pkg:maven/commons-logging/commons-logging@1.0.4",
	"/packages/TwilioNotifier.hpi:WEB-INF/lib/log4j-1.2.9.jar":                                                     "pkg:maven/log4j/log4j@1.2.9",
	"/packages/TwilioNotifier.hpi:WEB-INF/lib/sdk-3.0.jar:commons-codec:commons-codec":                             "pkg:maven/commons-codec/commons-codec@1.4",
	"/packages/TwilioNotifier.hpi:WEB-INF/lib/sdk-3.0.jar:commons-lang:commons-lang":                               "pkg:maven/commons-lang/commons-lang@2.5",
	"/packages/TwilioNotifier.hpi:WEB-INF/lib/sdk-3.0.jar:commons-logging:commons-logging":                         "pkg:maven/commons-logging/commons-logging@1.1.1",
	"/packages/TwilioNotifier.hpi:WEB-INF/lib/sdk-3.0.jar:org.apache.httpcomponents:httpclient":                    "pkg:maven/org.apache.httpcomponents/httpclient@4.1.1",
	"/packages/TwilioNotifier.hpi:WEB-INF/lib/sdk-3.0.jar:org.apache.httpcomponents:httpcore":                      "pkg:maven/org.apache.httpcomponents/httpcore@4.1",
	"/packages/TwilioNotifier.hpi:WEB-INF/lib/sdk-3.0.jar:org.easymock:easymock":                                   "pkg:maven/org.easymock/easymock@2.4",
	"/packages/undertow-core-2.3.4.Final.jar":                                                                      "pkg:maven/io.undertow/undertow-core@2.3.4.Final",
	"/packages/vaadin-client-8.11.3.jar":                                                                           "pkg:maven/com.vaadin/vaadin-client@8.11.3",
	"/packages/velocity-1.7.jar":                                                                                   "pkg:maven/org.apache.velocity/velocity@1.7",
	"/packages/vertx-web-4.3.7.jar":                                                                                "pkg:maven/io.vertx/vertx-web@4.3.7",
	"/packages/wicket-core-9.2.0.jar":                                                                              "pkg:maven/org.apache.wicket/wicket-core@9.2.0",
	"/packages/xalan-2.7.2.jar":                                                                                    "pkg:maven/xalan/xalan@2.7.2",
	"/packages/xmlpull-1.1.3.1.jar":                                                                                "pkg:maven/xmlpull/xmlpull@1.1.3.1",
	"/packages/xstream-1.4.11.1.jar":                                                                               "pkg:maven/com.thoughtworks.xstream/xstream@1.4.11.1",
	"/packages/xstream-1.4.19.jar":                                                                                 "pkg:maven/com.thoughtworks.xstream/xstream@1.4.19",
	"/packages/xwiki-commons-xml-15.1.jar":                                                                         "pkg:maven/org.xwiki.commons/xwiki-commons-xml@15.1",
	"/packages/xwiki-platform-oldcore-15.3.jar":                                                                    "pkg:maven/org.xwiki.platform/xwiki-platform-oldcore@15.3",
	"/packages/zookeeper-3.7.1.jar":                                                                                "pkg:maven/org.apache.zookeeper/zookeeper@3.7.1",
	"/packages/jruby-stdlib-9.1.15.0.jar:META-INF/jruby.home/lib/ruby/stdlib/readline.jar:jline:jline":             "pkg:maven/jline/jline@2.11",
	"/packages/jruby-stdlib-9.1.15.0.jar:META-INF/jruby.home/lib/ruby/stdlib/readline.jar:rubygems:jruby-readline": "pkg:maven/rubygems/jruby-readline@1.2.0",
}
