// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/FastJets.hh"
#include "Rivet/Projections/DISFinalState.hh"

namespace Rivet {


  /// @brief Multi-jet production in NC DIS
  class ZEUS_2005_I676091 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(ZEUS_2005_I676091);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {

      // The final-state particles are clustered in Breit frame
      // using FastJet with the kT algorithm and a jet-radius parameter of 1.
      const DISFinalState DISfs(DISFrame::BREIT);
      declare(DISfs, "fs");

      FastJets jets(DISfs, JetAlg::KT, 1.0);
      declare(jets, "jets");

      // Book histograms.
      book(_h["Et0"],    1, 1, 1);
      book(_h["Et1"],    2, 1, 1);
      book(_h["Et2"],    3, 1, 1);
      book(_h["Eta0"],   4, 1, 1);
      book(_h["Eta1"],   5, 1, 1);
      book(_h["Eta2"],   6, 1, 1);
      book(_h["2jetQ2"], 7, 1, 1);
      book(_h["3jetQ2"], 8, 1, 1);
      book(_ratio,       9, 1, 1);
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {

      // Lorentz invariant DIS quantities
      const DISKinematics& dis = apply<DISFinalState>(event, "fs").kinematics();
      if (dis.failed())  vetoEvent;

      const double Q2 = dis.Q2();
      const double y  = dis.y();

      // Kinematic cuts on virtuality and inelasticity.
      if ( !inRange(Q2, 10.*GeV2, 5000.*GeV2) )  vetoEvent;
      if ( !inRange(y, 0.04, 0.6) )              vetoEvent;

      // Lorentz boosts for Breit and lab frames.
      const LorentzTransform breitboost = dis.boostBreit();
      const LorentzTransform labboost = breitboost.inverse();

      // Retrieve clustered jets in Breit frame, sorted by pT.
      Jets alljets = apply<FastJets>(event, "jets").jets(Cuts::Et > 5*GeV, cmpMomByEt);

      // Cut on Pseurdorapidity in lab frame.
      // 1 if hadron in "conventional" +z direction, -1 if in -z.
      size_t jetcount = 3;
      Jets jets; vector<double> etas;
      const int orientation = dis.orientation();
      for (auto& jet : alljets) {
        jet.transformBy(labboost); // boost to lab frame
        if ( inRange(jet.eta()*orientation, -1., 2.5) ) {
          if (jetcount--)  etas.push_back(jet.eta()*orientation);
          jet.transformBy(breitboost);
          jets += jet;
        }
      }
      if (jets.size() < 2)  vetoEvent;

      if (jets.size() == 2) {
        if ( (jets[0].mom() + jets[1].mom()).mass() <= 25*GeV)  vetoEvent;
      }
      else if ((jets[0].mom() + jets[1].mom() + jets[2].mom()).mass() <= 25*GeV)  vetoEvent;

      _h["2jetQ2"]->fill(Q2/GeV2);

      if (jets.size() < 3)  vetoEvent;

      _h["3jetQ2"]->fill(Q2/GeV2);

      _h["Et0"]->fill(jets[0].Et()/GeV);
      _h["Et1"]->fill(jets[1].Et()/GeV);
      _h["Et2"]->fill(jets[2].Et()/GeV);

      std::sort(etas.begin(), etas.end(), std::greater<double>());
      _h["Eta0"]->fill(etas[0]);
      _h["Eta1"]->fill(etas[1]);
      _h["Eta2"]->fill(etas[2]);


    }

    /// Normalise histograms after the run
    void finalize() {

      scale(_h, crossSection()/picobarn/sumW());

      divide(_h["3jetQ2"], _h["2jetQ2"], _ratio);

    }

    /// @}

  private:

    /// @name Histograms
    /// @{
    map<string,Histo1DPtr> _h;
    Estimate1DPtr _ratio;
    /// @}
  };


  RIVET_DECLARE_PLUGIN(ZEUS_2005_I676091);

}
